"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_CORRELATED_LOGS_TOOL_ID = void 0;
exports.createGetCorrelatedLogsTool = createGetCorrelatedLogsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("../../utils/tool_schemas");
var _constants = require("./constants");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_CORRELATED_LOGS_TOOL_ID = exports.OBSERVABILITY_GET_CORRELATED_LOGS_TOOL_ID = 'observability.get_correlated_logs';
const getCorrelatedLogsSchema = _zod.z.object({
  ...(0, _tool_schemas.timeRangeSchemaOptional)(_constants.DEFAULT_TIME_RANGE),
  index: _zod.z.string().describe(_tool_schemas.indexDescription).optional(),
  logId: _zod.z.string().optional().describe('Optional ID of a specific log entry. If provided, the tool will fetch this log and find correlated logs based on its correlation identifier (e.g., trace.id). NOTE: When logId is provided, other filter parameters are ignored.'),
  kqlFilter: _zod.z.string().optional().describe('Optional KQL filter to narrow down logs. Example: "service.name: payment AND host.name: web-server-01". Ignored if logId is provided.'),
  errorLogsOnly: _zod.z.boolean().optional().describe('When true (default), only anchors on error logs (ERROR, WARN, FATAL, HTTP 5xx). Set to false to anchor on any log. For slow requests: kqlFilter="event.duration > 1000000", errorLogsOnly=false.'),
  correlationFields: _zod.z.array(_zod.z.string()).optional().describe('Optional list of field names to use for correlating logs. Use this when the user mentions a specific identifier (e.g., "group by session_id"). Overrides the default list of standard trace/request IDs. The first field in this list found with a value in an error log will be used to fetch the surrounding context.'),
  logSourceFields: _zod.z.array(_zod.z.string()).optional().describe('Optional list of fields to return for each log entry. If not provided, a default set of common Observability fields is returned. For a high-level overview, ["@timestamp", "message", "log.level"] is recommended.'),
  maxSequences: _zod.z.number().optional().describe('Optional maximum number of unique log sequences to return. Defaults to 10.'),
  maxLogsPerSequence: _zod.z.number().optional().describe('Optional maximum number of logs per sequence. Defaults to 200. Increase this to see a longer history of events surrounding the anchor.')
});
function createGetCorrelatedLogsTool({
  core,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_CORRELATED_LOGS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Retrieves complete log sequences around error events to understand what happened. By default, anchors on error logs (ERROR, WARN, FATAL, HTTP 5xx). Set errorLogsOnly=false to anchor on non-error events like slow requests.

When to use:
- Investigating WHY something failed or behaved unexpectedly
- Understanding the sequence of events leading to an error
- Following a request/transaction across services using correlation IDs (trace.id, request.id, etc.)

How it works:
1. Finds "anchor" logs (errors by default, or any log if errorLogsOnly=false)
2. Groups logs by correlation ID (trace.id, request.id, etc.)
3. Returns chronologically sorted sequences showing context before and after each anchor

Do NOT use for:
- High-level overview of log patterns (use get_log_categories)
- Analyzing log volume changes (use run_log_rate_analysis)`,
    schema: getCorrelatedLogsSchema,
    tags: ['observability', 'logs'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async (toolParams, {
      esClient
    }) => {
      const {
        start,
        end,
        kqlFilter,
        errorLogsOnly = true,
        index,
        correlationFields = _constants.DEFAULT_CORRELATION_IDENTIFIER_FIELDS,
        logId,
        logSourceFields = _constants.DEFAULT_LOG_SOURCE_FIELDS,
        maxSequences = 10,
        maxLogsPerSequence = 200
      } = toolParams;
      try {
        const {
          sequences
        } = await (0, _handler.getToolHandler)({
          core,
          logger,
          esClient,
          start,
          end,
          kqlFilter,
          errorLogsOnly,
          index,
          correlationFields,
          logId,
          logSourceFields,
          maxSequences,
          maxLogsPerSequence
        });
        if (sequences.length === 0) {
          const message = (0, _handler.getNoResultsMessage)({
            logId,
            kqlFilter,
            errorLogsOnly,
            correlationFields,
            start,
            end
          });
          return {
            results: [{
              type: _tool_result.ToolResultType.other,
              data: {
                sequences: [],
                message
              }
            }]
          };
        }
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              sequences
            }
          }]
        };
      } catch (error) {
        logger.error(`Error fetching errors and surrounding logs: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch errors and surrounding logs: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}