"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DiagnosticsAdapter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _monaco_imports = require("../monaco_imports");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const toDiagnostics = error => {
  return {
    ...error,
    severity: _monaco_imports.monaco.MarkerSeverity.Error
  };
};
class DiagnosticsAdapter {
  constructor(langId, worker) {
    (0, _defineProperty2.default)(this, "errors", {});
    (0, _defineProperty2.default)(this, "validation", new _rxjs.BehaviorSubject({
      isValid: true,
      isValidating: false,
      errors: []
    }));
    // To avoid stale validation data we keep track of the latest call to validate().
    (0, _defineProperty2.default)(this, "validateIdx", 0);
    (0, _defineProperty2.default)(this, "validation$", this.validation.asObservable());
    this.langId = langId;
    this.worker = worker;
    const onModelAdd = model => {
      let handle;
      if (model.getLanguageId() === this.langId) {
        model.onDidChangeContent(() => {
          // Do not validate if the language ID has changed
          if (model.getLanguageId() !== this.langId) {
            return;
          }
          const idx = ++this.validateIdx; // Disable any possible inflight validation
          clearTimeout(handle);

          // Reset the model markers if an empty string is provided on change
          if (model.getValue().trim() === '') {
            this.validation.next({
              isValid: true,
              isValidating: false,
              errors: []
            });
            return _monaco_imports.monaco.editor.setModelMarkers(model, this.langId, []);
          }
          this.validation.next({
            ...this.validation.value,
            isValidating: true
          });
          // Every time a new change is made, wait 500ms before validating
          handle = setTimeout(() => {
            this.validate(model.uri, idx);
          }, 500);
        });
        model.onDidChangeLanguage(({
          newLanguage
        }) => {
          // Reset the model markers if the language ID has changed and is no longer "painless"
          // Otherwise, re-validate
          if (newLanguage !== this.langId) {
            return _monaco_imports.monaco.editor.setModelMarkers(model, this.langId, []);
          } else {
            this.validate(model.uri, ++this.validateIdx);
          }
        });
        this.validation.next({
          ...this.validation.value,
          isValidating: true
        });
        this.validate(model.uri, ++this.validateIdx);
      }
    };
    _monaco_imports.monaco.editor.onDidCreateModel(onModelAdd);
    _monaco_imports.monaco.editor.getModels().forEach(onModelAdd);
  }
  async validate(resource, idx) {
    if (idx !== this.validateIdx) {
      return;
    }
    const worker = await this.worker(resource);
    const errorMarkers = await worker.getSyntaxErrors(resource.toString());
    if (idx !== this.validateIdx) {
      return;
    }
    if (errorMarkers) {
      const model = _monaco_imports.monaco.editor.getModel(resource);
      this.errors = {
        ...this.errors,
        [model.id]: errorMarkers
      };
      // Set the error markers and underline them with "Error" severity
      _monaco_imports.monaco.editor.setModelMarkers(model, this.langId, errorMarkers.map(toDiagnostics));
    }
    const isValid = errorMarkers === undefined || errorMarkers.length === 0;
    this.validation.next({
      isValidating: false,
      isValid,
      errors: errorMarkers !== null && errorMarkers !== void 0 ? errorMarkers : []
    });
  }
  getSyntaxErrors() {
    return this.errors;
  }
}
exports.DiagnosticsAdapter = DiagnosticsAdapter;