"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AnnotationsStateService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _mlDatePicker = require("@kbn/ml-date-picker");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _state_service = require("../services/state_service");
var _search = require("../../../common/constants/search");
var _explorer_utils = require("./explorer_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Dedicated state service for annotations.
 * - overallAnnotations$ drives the swimlane overlay
 * - annotationsTable$ drives the Annotations panel
 */
class AnnotationsStateService extends _state_service.StateService {
  constructor(mlApi, timefilter, anomalyExplorerCommonStateService, anomalyTimelineStateService) {
    super();
    (0, _defineProperty2.default)(this, "_overallAnnotations$", new _rxjs.BehaviorSubject({
      annotationsData: [],
      error: undefined
    }));
    (0, _defineProperty2.default)(this, "_annotationsTable$", new _rxjs.BehaviorSubject({
      annotationsData: [],
      totalCount: 0,
      error: undefined
    }));
    (0, _defineProperty2.default)(this, "_timeBounds$", void 0);
    (0, _defineProperty2.default)(this, "_refreshSubject$", void 0);
    this.mlApi = mlApi;
    this.timefilter = timefilter;
    this.anomalyExplorerCommonStateService = anomalyExplorerCommonStateService;
    this.anomalyTimelineStateService = anomalyTimelineStateService;
    this._timeBounds$ = this.timefilter.getTimeUpdate$().pipe((0, _rxjs.startWith)(null), (0, _rxjs.map)(() => this.timefilter.getBounds()));
    this._refreshSubject$ = _mlDatePicker.mlTimefilterRefresh$.pipe((0, _rxjs.startWith)({
      lastRefresh: 0
    }));
    this._init();
  }
  get overallAnnotations$() {
    return this._overallAnnotations$.asObservable();
  }
  get annotationsTable$() {
    return this._annotationsTable$.asObservable();
  }
  get overallAnnotations() {
    return this._overallAnnotations$.getValue();
  }
  get annotationsTable() {
    return this._annotationsTable$.getValue();
  }
  _initSubscriptions() {
    const subscription = new _rxjs.Subscription();

    // Overall annotations react to selected jobs, time bounds and manual refreshes
    subscription.add((0, _rxjs.combineLatest)([this.anomalyExplorerCommonStateService.selectedJobs$, this._timeBounds$, this._refreshSubject$]).pipe((0, _rxjs.switchMap)(([selectedJobs, bounds]) => {
      return this._loadOverallAnnotations(selectedJobs, bounds);
    })).subscribe(this._overallAnnotations$));

    // Annotations table reacts to selected cells, selected jobs, time bounds and refreshes
    subscription.add((0, _rxjs.combineLatest)([this.anomalyTimelineStateService.getSelectedCells$(), this.anomalyExplorerCommonStateService.selectedJobs$, this._timeBounds$, this._refreshSubject$]).pipe((0, _rxjs.switchMap)(([selectedCells, selectedJobs, bounds]) => (0, _explorer_utils.loadAnnotationsTableData)(this.mlApi, selectedCells, selectedJobs, bounds))).subscribe(this._annotationsTable$));
    return subscription;
  }
  _loadOverallAnnotations(selectedJobs, bounds) {
    const jobIds = selectedJobs.map(d => d.id);
    const timeRange = (0, _explorer_utils.getSelectionTimeRange)(undefined, bounds);
    return new Promise(resolve => {
      (0, _rxjs.lastValueFrom)(this.mlApi.annotations.getAnnotations$({
        jobIds,
        earliestMs: timeRange.earliestMs,
        latestMs: timeRange.latestMs,
        maxAnnotations: _search.ANNOTATIONS_TABLE_DEFAULT_QUERY_SIZE
      })).then(resp => {
        if (resp.error !== undefined || resp.annotations === undefined) {
          const errorMessage = (0, _mlErrorUtils.extractErrorMessage)(resp.error);
          return resolve({
            annotationsData: [],
            error: errorMessage !== '' ? errorMessage : undefined
          });
        }
        const annotationsData = [];
        jobIds.forEach(jobId => {
          const jobAnnotations = resp.annotations[jobId];
          if (jobAnnotations !== undefined) {
            annotationsData.push(...jobAnnotations);
          }
        });
        return resolve({
          annotationsData: annotationsData.sort((a, b) => {
            return a.timestamp - b.timestamp;
          }).map((d, i) => {
            d.key = (i + 1).toString();
            return d;
          })
        });
      }).catch(resp => {
        const errorMessage = (0, _mlErrorUtils.extractErrorMessage)(resp);
        return resolve({
          annotationsData: [],
          error: errorMessage !== '' ? errorMessage : undefined
        });
      });
    });
  }
}
exports.AnnotationsStateService = AnnotationsStateService;