"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initMetrics = initMetrics;
var _lodash = require("lodash");
var _grpcJs = require("@grpc/grpc-js");
var _exporterMetricsOtlpGrpc = require("@opentelemetry/exporter-metrics-otlp-grpc");
var _exporterMetricsOtlpHttp = require("@opentelemetry/exporter-metrics-otlp-http");
var _sdk = require("@elastic/opentelemetry-node/sdk");
var _std = require("@kbn/std");
var _cleanupBeforeExit = require("@kbn/cleanup-before-exit");
var _moment = require("moment");
var _prometheus_exporter = require("./prometheus_exporter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Options to the initMetrics method
 */

/**
 * Initialize the OpenTelemetry meter provider
 * @param initMetricsOptions {@link InitMetricsOptions}
 */
function initMetrics(initMetricsOptions) {
  var _metricsConfig$timeou;
  const {
    resource,
    metricsConfig,
    monitoringCollectionConfig
  } = initMetricsOptions;
  const globalExportIntervalMillis = metricsConfig.interval.asMilliseconds();
  const globalExportTimeoutMillis = (_metricsConfig$timeou = metricsConfig.timeout) === null || _metricsConfig$timeou === void 0 ? void 0 : _metricsConfig$timeou.asMilliseconds();
  const readers = [];

  // OTel requires all readers to be initialized when initializing the SDK.
  // If the monitoring collection plugin requires the prometheus exporter, we need to initialize it now.
  if (monitoringCollectionConfig.enabled && monitoringCollectionConfig.opentelemetry.metrics.prometheus.enabled) {
    readers.push(_prometheus_exporter.PrometheusExporter.get());
  }
  const exporters = metricsConfig.enabled ? (0, _lodash.castArray)(metricsConfig.exporters) : [];
  if (monitoringCollectionConfig.enabled) {
    var _process$env$OTEL_EXP;
    // For BWC reasons, we want to support the OTLP configuration present in the monitoring collection plugin.
    const otlpConfig = monitoringCollectionConfig.opentelemetry.metrics.otlp;
    const {
      url = (_process$env$OTEL_EXP = process.env.OTEL_EXPORTER_OTLP_METRICS_ENDPOINT) !== null && _process$env$OTEL_EXP !== void 0 ? _process$env$OTEL_EXP : process.env.OTEL_EXPORTER_OTLP_ENDPOINT,
      headers,
      exportIntervalMillis
    } = otlpConfig;
    if (url) {
      var _ref;
      const temporalityPreference = (_ref = process.env.OTEL_EXPORTER_OTLP_METRICS_TEMPORALITY_PREFERENCE) !== null && _ref !== void 0 ? _ref : 'cumulative';

      // We just need to push it as another grpc config
      exporters.push({
        grpc: {
          url,
          headers,
          exportInterval: (0, _moment.duration)(exportIntervalMillis, 'ms'),
          temporalityPreference
        }
      });
    }
  }
  readers.push(...exporters.map(exporterConfig => {
    var _variant$value$export, _variant$value$export2;
    const variant = (0, _std.fromExternalVariant)(exporterConfig);
    const commonConfig = {
      temporalityPreference: variant.value.temporalityPreference === 'delta' ? _sdk.metrics.AggregationTemporality.DELTA : _sdk.metrics.AggregationTemporality.CUMULATIVE
    };
    let exporter;
    switch (variant.type) {
      case 'grpc':
        {
          const metadata = new _grpcJs.Metadata();
          Object.entries(variant.value.headers || {}).forEach(([key, value]) => {
            metadata.add(key, value);
          });
          exporter = new _exporterMetricsOtlpGrpc.OTLPMetricExporter({
            ...commonConfig,
            metadata,
            url: variant.value.url
          });
          break;
        }
      case 'http':
        exporter = new _exporterMetricsOtlpHttp.OTLPMetricExporter({
          ...commonConfig,
          headers: variant.value.headers,
          url: variant.value.url
        });
        break;
    }
    const exportInterval = (_variant$value$export = variant.value.exportInterval) !== null && _variant$value$export !== void 0 ? _variant$value$export : globalExportIntervalMillis;
    const exportIntervalMillis = typeof exportInterval === 'number' ? exportInterval : exportInterval.asMilliseconds();

    // If the exporter's export timeout is not provided, use the global export timeout only if it is less than the export interval (the client fails if the timeout is higher than the interval).
    // Otherwise, we use the export interval.
    let exportTimeoutMillis = (_variant$value$export2 = variant.value.exportTimeout) === null || _variant$value$export2 === void 0 ? void 0 : _variant$value$export2.asMilliseconds();
    if (typeof exportTimeoutMillis !== 'number') {
      if (globalExportTimeoutMillis && globalExportTimeoutMillis <= exportIntervalMillis) {
        exportTimeoutMillis = globalExportTimeoutMillis;
      } else {
        exportTimeoutMillis = exportIntervalMillis;
      }
    }
    return new _sdk.metrics.PeriodicExportingMetricReader({
      exporter,
      exportIntervalMillis,
      exportTimeoutMillis
    });
  }));
  const meterProvider = new _sdk.metrics.MeterProvider({
    readers,
    resource,
    views: []
  });
  _sdk.api.metrics.setGlobalMeterProvider(meterProvider);
  (0, _cleanupBeforeExit.cleanupBeforeExit)(() => meterProvider.shutdown());
}