"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMetricIndices = exports.useInfrastructureNodeMetrics = void 0;
var _datemath = require("@kbn/datemath");
var _react = require("react");
var _use_tracked_promise = require("../../../../utils/use_tracked_promise");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useMetricIndices = ({
  metricsClient
}) => {
  const [metricIndices, setMetricIndices] = (0, _react.useState)(undefined);
  const [metricIndicesRequest, getMetricIndices] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: () => {
      return metricsClient.metricsIndices();
    },
    onResolve: response => {
      if (response) {
        setMetricIndices(response);
      }
    }
  }, [metricsClient]);
  (0, _react.useEffect)(() => {
    getMetricIndices();
  }, [getMetricIndices, metricsClient]);
  const hasFailedLoading = metricIndicesRequest.state === 'rejected';
  const isUninitialized = metricIndicesRequest.state === 'uninitialized';
  const isLoading = metricIndicesRequest.state === 'pending';
  return {
    isLoading,
    isUninitialized,
    errorMessage: hasFailedLoading ? `${metricIndicesRequest.value}` : undefined,
    metricIndicesExist: metricIndices === null || metricIndices === void 0 ? void 0 : metricIndices.metricIndicesExist,
    metricIndices: metricIndices === null || metricIndices === void 0 ? void 0 : metricIndices.metricIndices
  };
};
exports.useMetricIndices = useMetricIndices;
const NODE_COUNT_LIMIT = 10000;
const TOTAL_NODES_LIMIT = 100;
const TABLE_PAGE_SIZE = 10;
const nullData = {
  series: [],
  pageInfo: {
    afterKey: null,
    total: -1
  }
};
const useInfrastructureNodeMetrics = options => {
  const {
    metricsExplorerOptions,
    timerange,
    transform,
    sortState,
    currentPageIndex,
    metricsClient
  } = options;
  const [transformedNodes, setTransformedNodes] = (0, _react.useState)([]);
  const {
    metricIndices,
    metricIndicesExist,
    isLoading: metricIndicesLoading,
    errorMessage: metricIndicesError
  } = useMetricIndices({
    metricsClient
  });
  const timerangeWithInterval = useTimerangeWithInterval(timerange);
  const [fetchNodesRequest, fetchNodes] = (0, _use_tracked_promise.useTrackedPromise)({
    createPromise: async () => {
      if (!metricIndices) {
        return Promise.resolve(nullData);
      }
      const request = {
        metrics: metricsExplorerOptions.metrics,
        groupBy: metricsExplorerOptions.groupBy,
        limit: NODE_COUNT_LIMIT,
        indexPattern: metricIndices,
        kuery: metricsExplorerOptions.kuery,
        timerange: timerangeWithInterval
      };
      return metricsClient.metricsExplorer(request);
    },
    onResolve: response => {
      setTransformedNodes(response.series.map(transform));
    },
    cancelPreviousOn: 'creation'
  }, [metricIndices, metricsExplorerOptions, timerangeWithInterval]);
  const isLoadingNodes = fetchNodesRequest.state === 'pending' || fetchNodesRequest.state === 'uninitialized';
  const isLoading = metricIndicesLoading || isLoadingNodes;
  const errors = (0, _react.useMemo)(() => [...(metricIndicesError ? [wrapAsError(metricIndicesError)] : []), ...(fetchNodesRequest.state === 'rejected' ? [wrapAsError(fetchNodesRequest.value)] : [])], [fetchNodesRequest, metricIndicesError]);
  (0, _react.useEffect)(() => {
    fetchNodes();
  }, [fetchNodes]);
  const sortedNodes = (0, _react.useMemo)(() => {
    return [...transformedNodes].sort(makeSortNodes(sortState));
  }, [transformedNodes, sortState]);
  const top100Nodes = (0, _react.useMemo)(() => {
    return sortedNodes.slice(0, TOTAL_NODES_LIMIT);
  }, [sortedNodes]);
  const nodes = (0, _react.useMemo)(() => {
    const pageStartIndex = currentPageIndex * TABLE_PAGE_SIZE;
    const pageEndIndex = pageStartIndex + TABLE_PAGE_SIZE;
    return top100Nodes.slice(pageStartIndex, pageEndIndex);
  }, [top100Nodes, currentPageIndex]);
  const pageCount = (0, _react.useMemo)(() => Math.ceil(top100Nodes.length / TABLE_PAGE_SIZE), [top100Nodes]);
  const data = (0, _react.useMemo)(() => errors.length > 0 ? {
    state: 'error',
    errors
  } : metricIndicesExist == null ? {
    state: 'unknown'
  } : !metricIndicesExist ? {
    state: 'no-indices'
  } : nodes.length <= 0 ? {
    state: 'empty-indices'
  } : {
    state: 'data',
    currentPageIndex,
    pageCount,
    rows: nodes
  }, [currentPageIndex, errors, metricIndicesExist, nodes, pageCount]);
  return {
    isLoading,
    data
  };
};
exports.useInfrastructureNodeMetrics = useInfrastructureNodeMetrics;
function useTimerangeWithInterval(timerange) {
  return (0, _react.useMemo)(() => {
    const from = (0, _datemath.parse)(timerange.from);
    const to = (0, _datemath.parse)(timerange.to);
    if (!from || !to) {
      throw new Error('Could not parse timerange');
    }
    return {
      from: from.valueOf(),
      to: to.valueOf(),
      interval: 'modules'
    };
  }, [timerange]);
}
function makeSortNodes(sortState) {
  return (nodeA, nodeB) => {
    const nodeAValue = nodeA[sortState.field];
    const nodeBValue = nodeB[sortState.field];
    if (sortState.direction === 'asc') {
      return sortAscending(nodeAValue, nodeBValue);
    }
    return sortDescending(nodeAValue, nodeBValue);
  };
}
function sortAscending(nodeAValue, nodeBValue) {
  if (nodeAValue === null) {
    return -1;
  } else if (nodeBValue === null) {
    return 1;
  }
  if (typeof nodeAValue === 'string' && typeof nodeBValue === 'string') {
    return nodeAValue.localeCompare(nodeBValue);
  }
  if (typeof nodeAValue === 'number' && typeof nodeBValue === 'number') {
    return nodeAValue - nodeBValue;
  }
  return 0;
}
function sortDescending(nodeAValue, nodeBValue) {
  if (nodeAValue === null) {
    return 1;
  } else if (nodeBValue === null) {
    return -1;
  }
  if (typeof nodeAValue === 'string' && typeof nodeBValue === 'string') {
    return nodeBValue.localeCompare(nodeAValue);
  }
  if (typeof nodeAValue === 'number' && typeof nodeBValue === 'number') {
    return nodeBValue - nodeAValue;
  }
  return 0;
}
const wrapAsError = value => value instanceof Error ? value : new Error(`${value}`);