"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metricByField = void 0;
exports.useContainerMetricsTable = useContainerMetricsTable;
var _react = require("react");
var _shared = require("../shared");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const containerMetricsQueryConfig = {
  sourceFilter: `event.dataset: "kubernetes.container"`,
  groupByField: 'container.id',
  metricsMap: {
    'kubernetes.container.cpu.usage.limit.pct': {
      aggregation: 'avg',
      field: 'kubernetes.container.cpu.usage.limit.pct'
    },
    'kubernetes.container.memory.usage.bytes': {
      aggregation: 'avg',
      field: 'kubernetes.container.memory.usage.bytes'
    }
  }
};
const metricByField = exports.metricByField = (0, _shared.createMetricByFieldLookup)(containerMetricsQueryConfig.metricsMap);
const unpackMetric = (0, _shared.makeUnpackMetric)(metricByField);
function useContainerMetricsTable({
  timerange,
  kuery,
  metricsClient
}) {
  const [currentPageIndex, setCurrentPageIndex] = (0, _react.useState)(0);
  const [sortState, setSortState] = (0, _react.useState)({
    field: 'averageCpuUsagePercent',
    direction: 'desc'
  });
  const {
    options: containerMetricsOptions
  } = (0, _react.useMemo)(() => (0, _shared.metricsToApiOptions)(containerMetricsQueryConfig, kuery), [kuery]);
  const {
    data,
    isLoading
  } = (0, _shared.useInfrastructureNodeMetrics)({
    metricsExplorerOptions: containerMetricsOptions,
    timerange,
    transform: seriesToContainerNodeMetricsRow,
    sortState,
    currentPageIndex,
    metricsClient
  });
  return {
    data,
    isLoading,
    setCurrentPageIndex,
    setSortState,
    sortState,
    timerange
  };
}
function seriesToContainerNodeMetricsRow(series) {
  if (series.rows.length === 0) {
    return rowWithoutMetrics(series.id);
  }
  return {
    id: series.id,
    ...calculateMetricAverages(series.rows)
  };
}
function rowWithoutMetrics(id) {
  return {
    id,
    averageCpuUsagePercent: null,
    averageMemoryUsageMegabytes: null
  };
}
function calculateMetricAverages(rows) {
  const {
    averageCpuUsagePercentValues,
    averageMemoryUsageMegabytesValues
  } = collectMetricValues(rows);
  let averageCpuUsagePercent = null;
  if (averageCpuUsagePercentValues.length !== 0) {
    averageCpuUsagePercent = (0, _shared.scaleUpPercentage)((0, _shared.averageOfValues)(averageCpuUsagePercentValues));
  }
  let averageMemoryUsageMegabytes = null;
  if (averageMemoryUsageMegabytesValues.length !== 0) {
    const averageInBytes = (0, _shared.averageOfValues)(averageMemoryUsageMegabytesValues);
    const bytesPerMegabyte = 1000000;
    averageMemoryUsageMegabytes = Math.floor(averageInBytes / bytesPerMegabyte);
  }
  return {
    averageCpuUsagePercent,
    averageMemoryUsageMegabytes
  };
}
function collectMetricValues(rows) {
  const averageCpuUsagePercentValues = [];
  const averageMemoryUsageMegabytesValues = [];
  rows.forEach(row => {
    const {
      averageCpuUsagePercent,
      averageMemoryUsageMegabytes
    } = unpackMetrics(row);
    if (averageCpuUsagePercent !== null) {
      averageCpuUsagePercentValues.push(averageCpuUsagePercent);
    }
    if (averageMemoryUsageMegabytes !== null) {
      averageMemoryUsageMegabytesValues.push(averageMemoryUsageMegabytes);
    }
  });
  return {
    averageCpuUsagePercentValues,
    averageMemoryUsageMegabytesValues
  };
}
function unpackMetrics(row) {
  return {
    averageCpuUsagePercent: unpackMetric(row, 'kubernetes.container.cpu.usage.limit.pct'),
    averageMemoryUsageMegabytes: unpackMetric(row, 'kubernetes.container.memory.usage.bytes')
  };
}