"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LogsSharedKibanaLogEntriesAdapter = void 0;
var _lodash = require("lodash");
var _log_entries_domain = require("../../domains/log_entries_domain");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TIMESTAMP_FORMAT = 'epoch_millis';
class LogsSharedKibanaLogEntriesAdapter {
  constructor(framework) {
    this.framework = framework;
  }
  async getLogEntries(requestContext, resolvedLogView, fields, params) {
    var _params$size;
    const {
      startTimestamp,
      endTimestamp,
      query,
      cursor,
      highlightTerm
    } = params;
    const size = (_params$size = params.size) !== null && _params$size !== void 0 ? _params$size : _log_entries_domain.LOG_ENTRIES_PAGE_SIZE;
    const {
      sortDirection,
      searchAfterClause
    } = processCursor(cursor);
    const highlightQuery = createHighlightQuery(highlightTerm, fields);
    const highlightClause = highlightQuery ? {
      highlight: {
        boundary_scanner: 'word',
        fields: fields.reduce((highlightFieldConfigs, fieldName) => {
          highlightFieldConfigs[fieldName] = {};
          return highlightFieldConfigs;
        }, {}),
        fragment_size: 1,
        number_of_fragments: 100,
        post_tags: [''],
        pre_tags: [''],
        highlight_query: highlightQuery
      }
    } : {};
    const sort = {
      [_constants.TIMESTAMP_FIELD]: {
        order: sortDirection,
        format: 'strict_date_optional_time_nanos',
        numeric_type: 'date_nanos'
      },
      [_constants.TIEBREAKER_FIELD]: sortDirection
    };
    const esQuery = {
      allow_no_indices: true,
      index: resolvedLogView.indices,
      ignore_unavailable: true,
      body: {
        size: size + 1,
        // Extra one to test if it has more before or after
        track_total_hits: false,
        _source: false,
        fields,
        query: {
          bool: {
            filter: [...createFilterClauses(query, highlightQuery), {
              range: {
                [_constants.TIMESTAMP_FIELD]: {
                  gte: startTimestamp,
                  lte: endTimestamp,
                  format: TIMESTAMP_FORMAT
                }
              }
            }]
          }
        },
        runtime_mappings: resolvedLogView.runtimeMappings,
        sort,
        ...highlightClause,
        ...searchAfterClause
      }
    };
    const esResult = await this.framework.callWithRequest(requestContext, 'search', esQuery);
    const hits = esResult.hits.hits;
    const hasMore = hits.length > size;
    if (hasMore) {
      hits.pop();
    }
    if (sortDirection === 'desc') {
      hits.reverse();
    }
    return {
      documents: mapHitsToLogEntryDocuments(hits, fields),
      hasMoreBefore: sortDirection === 'desc' ? hasMore : undefined,
      hasMoreAfter: sortDirection === 'asc' ? hasMore : undefined
    };
  }
}
exports.LogsSharedKibanaLogEntriesAdapter = LogsSharedKibanaLogEntriesAdapter;
function mapHitsToLogEntryDocuments(hits, fields) {
  return hits.map(hit => {
    const logFields = fields.reduce((flattenedFields, field) => field in hit.fields ? {
      ...flattenedFields,
      [field]: hit.fields[field]
    } : flattenedFields, {});
    return {
      id: hit._id,
      index: hit._index,
      cursor: {
        time: hit.sort[0],
        tiebreaker: hit.sort[1]
      },
      fields: logFields,
      highlights: hit.highlight || {}
    };
  });
}
const createHighlightQuery = (highlightTerm, fields) => {
  if (highlightTerm) {
    return {
      multi_match: {
        fields,
        lenient: true,
        query: highlightTerm,
        type: 'phrase'
      }
    };
  }
};
const createFilterClauses = (filterQuery, highlightQuery) => {
  if (filterQuery && highlightQuery) {
    return [{
      bool: {
        filter: [filterQuery, highlightQuery]
      }
    }];
  }
  return (0, _lodash.compact)([filterQuery, highlightQuery]);
};
function processCursor(cursor) {
  if (cursor) {
    if ('before' in cursor) {
      return {
        sortDirection: 'desc',
        searchAfterClause: cursor.before !== 'last' ? {
          search_after: [cursor.before.time, cursor.before.tiebreaker]
        } : {}
      };
    } else if (cursor.after !== 'first') {
      return {
        sortDirection: 'asc',
        searchAfterClause: {
          search_after: [cursor.after.time, cursor.after.tiebreaker]
        }
      };
    }
  }
  return {
    sortDirection: 'asc',
    searchAfterClause: {}
  };
}