"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LockManager = exports.LockAcquisitionError = void 0;
exports.getLock = getLock;
exports.isLockAcquisitionError = isLockAcquisitionError;
exports.rerunSetupIndexAsset = rerunSetupIndexAsset;
exports.runSetupIndexAssetOnce = void 0;
exports.withLock = withLock;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticsearch = require("@elastic/elasticsearch");
var _uuid = require("uuid");
var _prettyMs = _interopRequireDefault(require("pretty-ms"));
var _moment = require("moment");
var _setup_lock_manager_index = require("./setup_lock_manager_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// eslint-disable-next-line max-classes-per-file

// The index assets should only be set up once
let runLockManagerSetupSuccessfully = false;
const runSetupIndexAssetOnce = async (esClient, logger) => {
  if (runLockManagerSetupSuccessfully) {
    return;
  }
  await (0, _setup_lock_manager_index.setupLockManagerIndex)(esClient, logger);
  runLockManagerSetupSuccessfully = true;
};

// For testing purposes, we need to be able to set it up every time
exports.runSetupIndexAssetOnce = runSetupIndexAssetOnce;
function rerunSetupIndexAsset() {
  runLockManagerSetupSuccessfully = false;
}
class LockManager {
  constructor(lockId, esClient, logger) {
    (0, _defineProperty2.default)(this, "token", (0, _uuid.v4)());
    this.lockId = lockId;
    this.esClient = esClient;
    this.logger = logger;
  }

  /**
   * Attempts to acquire a lock by creating a document with the given lockId.
   * If the lock exists and is expired, it will be released and acquisition retried.
   */
  async acquire({
    metadata = {},
    ttl = (0, _moment.duration)(30, 'seconds').asMilliseconds()
  } = {}) {
    let response;
    await runSetupIndexAssetOnce(this.esClient, this.logger);
    this.token = (0, _uuid.v4)();
    try {
      response = await this.esClient.update({
        index: _setup_lock_manager_index.LOCKS_CONCRETE_INDEX_NAME,
        id: this.lockId,
        scripted_upsert: true,
        script: {
          lang: 'painless',
          source: `
              // Get the current time on the ES server.
              long now = System.currentTimeMillis();
              
              // If creating the document, or if the lock is expired,
              // or if the current document is owned by the same token, then update it.
              if (ctx.op == 'create' ||
                  Instant.parse(ctx._source.expiresAt).toEpochMilli() < now ||
                  ctx._source.token == params.token) {
                def instantNow = Instant.ofEpochMilli(now);
                ctx._source.createdAt = instantNow.toString();
                ctx._source.expiresAt = instantNow.plusMillis(params.ttl).toString();
                ctx._source.metadata = params.metadata;
                ctx._source.token = params.token;
              } else {
                ctx.op = 'noop';
              }
            `,
          params: {
            ttl,
            token: this.token,
            metadata
          }
        },
        // @ts-expect-error
        upsert: {}
      }, {
        retryOnTimeout: true,
        maxRetries: 3
      });
    } catch (e) {
      if (isVersionConflictException(e)) {
        this.logger.debug(`Lock "${this.lockId}" already held (version conflict)`);
        return false;
      }
      this.logger.error(`Failed to acquire lock "${this.lockId}": ${e.message}`);
      return false;
    }
    switch (response.result) {
      case 'created':
        {
          this.logger.debug(`Lock "${this.lockId}" with token = ${this.token} acquired with ttl = ${(0, _prettyMs.default)(ttl)}`);
          return true;
        }
      case 'updated':
        {
          this.logger.debug(`Lock "${this.lockId}" was expired and re-acquired with ttl = ${(0, _prettyMs.default)(ttl)} and token = ${this.token}`);
          return true;
        }
      case 'noop':
        {
          this.logger.debug(`Lock "${this.lockId}" with token = ${this.token} could not be acquired. It is already held`);
          return false;
        }
    }
    this.logger.warn(`Unexpected response: ${response.result}`);
    return false;
  }

  /**
   * Releases the lock by deleting the document with the given lockId and token
   */
  async release() {
    let response;
    try {
      response = await this.esClient.update({
        index: _setup_lock_manager_index.LOCKS_CONCRETE_INDEX_NAME,
        id: this.lockId,
        scripted_upsert: false,
        script: {
          lang: 'painless',
          source: `
            if (ctx._source.token == params.token) {
              ctx.op = 'delete';
            } else {
              ctx.op = 'noop';
            }
          `,
          params: {
            token: this.token
          }
        }
      }, {
        retryOnTimeout: true,
        maxRetries: 3
      });
    } catch (error) {
      if (isDocumentMissingException(error)) {
        this.logger.debug(`Lock "${this.lockId}" already released.`);
        return false;
      }
      this.logger.error(`Failed to release lock "${this.lockId}": ${error.message}`);
      throw error;
    }
    switch (response.result) {
      case 'deleted':
        this.logger.debug(`Lock "${this.lockId}" released with token ${this.token}.`);
        return true;
      case 'noop':
        this.logger.warn(`Lock "${this.lockId}" with token = ${this.token} could not be released. Token does not match.`);
        return false;
    }
    this.logger.warn(`Unexpected response: ${response.result}`);
    return false;
  }

  /**
   * Retrieves the lock document for a given lockId.
   * If the lock is expired, it will not be returned
   */
  async get() {
    var _result$_source;
    const result = await this.esClient.get({
      index: _setup_lock_manager_index.LOCKS_CONCRETE_INDEX_NAME,
      id: this.lockId
    }, {
      ignore: [404]
    });
    if (!result._source) {
      return undefined;
    }
    const isExpired = new Date((_result$_source = result._source) === null || _result$_source === void 0 ? void 0 : _result$_source.expiresAt).getTime() < Date.now();
    if (isExpired) {
      return undefined;
    }
    return result._source;
  }
  async extendTtl(ttl) {
    try {
      await this.esClient.update({
        index: _setup_lock_manager_index.LOCKS_CONCRETE_INDEX_NAME,
        id: this.lockId,
        script: {
          lang: 'painless',
          source: `
          if (ctx._source.token == params.token) {
            long now = System.currentTimeMillis();
            ctx._source.expiresAt = Instant.ofEpochMilli(now + params.ttl).toString();
          } else {
            ctx.op = 'noop';
          }`,
          params: {
            ttl,
            token: this.token
          }
        }
      });
      this.logger.debug(`Lock "${this.lockId}" extended ttl with ${(0, _prettyMs.default)(ttl)}.`);
      return true;
    } catch (error) {
      if (isVersionConflictException(error) || isDocumentMissingException(error)) {
        this.logger.debug(`Lock "${this.lockId}" was released concurrently. Not extending TTL.`);
        return false;
      }
      this.logger.error(`Failed to extend lock "${this.lockId}": ${error.message}`);
      this.logger.debug(error);
      return false;
    }
  }
}
exports.LockManager = LockManager;
async function getLock({
  esClient,
  logger,
  lockId
}) {
  const lockManager = new LockManager(lockId, esClient, logger);
  return lockManager.get();
}
function isLockAcquisitionError(error) {
  return error instanceof LockAcquisitionError;
}
async function withLock({
  esClient,
  logger,
  lockId,
  metadata,
  ttl = (0, _moment.duration)(30, 'seconds').asMilliseconds()
}, callback) {
  const lockManager = new LockManager(lockId, esClient, logger);
  const acquired = await lockManager.acquire({
    metadata,
    ttl
  });
  if (!acquired) {
    logger.debug(`Lock "${lockId}" not acquired. Exiting.`);
    throw new LockAcquisitionError(`Lock "${lockId}" not acquired`);
  }

  // extend the ttl periodically
  const extendInterval = Math.floor(ttl / 4);
  logger.debug(`Extending TTL for lock "${lockId}" every ${(0, _prettyMs.default)(extendInterval)}`);
  let extendTTlPromise = Promise.resolve(true);
  const intervalId = setInterval(() => {
    // wait for the previous extendTtl request to finish before sending the next one. This is to avoid flooding ES with extendTtl requests in cases where ES is slow to respond.
    extendTTlPromise = extendTTlPromise.then(() => lockManager.extendTtl(ttl)).catch(err => {
      logger.error(`Failed to extend lock "${lockId}":`, err);
      return false;
    });
  }, extendInterval);
  try {
    return await callback();
  } finally {
    try {
      clearInterval(intervalId);
      await extendTTlPromise;
      await lockManager.release();
    } catch (error) {
      logger.error(`Failed to release lock "${lockId}" in withLock: ${error.message}`);
      logger.debug(error);
    }
  }
}
function isVersionConflictException(e) {
  var _e$body, _e$body$error;
  return e instanceof _elasticsearch.errors.ResponseError && ((_e$body = e.body) === null || _e$body === void 0 ? void 0 : (_e$body$error = _e$body.error) === null || _e$body$error === void 0 ? void 0 : _e$body$error.type) === 'version_conflict_engine_exception';
}
function isDocumentMissingException(e) {
  var _e$body2, _e$body2$error;
  return e instanceof _elasticsearch.errors.ResponseError && ((_e$body2 = e.body) === null || _e$body2 === void 0 ? void 0 : (_e$body2$error = _e$body2.error) === null || _e$body2$error === void 0 ? void 0 : _e$body2$error.type) === 'document_missing_exception';
}
class LockAcquisitionError extends Error {
  constructor(message) {
    super(message);
    this.name = 'LockAcquisitionError';
  }
}
exports.LockAcquisitionError = LockAcquisitionError;