"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEndpointListRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _api = require("@kbn/securitysolution-endpoint-exceptions-common/api");
var _constants = require("@kbn/security-solution-features/constants");
var _utils = require("./utils");
var _get_exception_list_client = require("./utils/get_exception_list_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This creates the endpoint list if it does not exist. If it does exist,
 * this will conflict but continue. This is intended to be as fast as possible so it tries
 * each and every time it is called to create the endpoint_list and just ignores any
 * conflict so at worse case only one round trip happens per API call. If any error other than conflict
 * happens this will return that error. If the list already exists this will return an empty
 * object.
 * @param router The router to use.
 */
const createEndpointListRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _securitysolutionListConstants.ENDPOINT_LIST_URL,
    security: {
      authz: {
        requiredPrivileges: [_constants.LISTS_API_ALL]
      }
    }
  }).addVersion({
    validate: false,
    version: '2023-10-31'
  }, async (context, _, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const exceptionLists = await (0, _get_exception_list_client.getExceptionListClient)(context);
      const createdList = await exceptionLists.createEndpointList();

      // We always return ok on a create  endpoint list route but with an empty body as
      // an additional fetch of the full list would be slower and the UI has everything hard coded
      // within it to get the list if it needs details about it. Our goal is to be as fast as possible
      // and block the least amount of time with this route since it could end up in various parts of the
      // stack at some point such as repeatedly being called by endpoint agents.
      return response.ok({
        body: _api.CreateEndpointListResponse.parse(createdList !== null && createdList !== void 0 ? createdList : {})
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.createEndpointListRoute = createEndpointListRoute;