"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LensConfigBuilder = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _charts = require("./charts");
var _metric = require("./transforms/charts/metric");
var _legacy_metric = require("./transforms/charts/legacy_metric");
var _xy = require("./transforms/charts/xy");
var _gauge = require("./transforms/charts/gauge");
var _heatmap = require("./transforms/charts/heatmap");
var _tagcloud = require("./transforms/charts/tagcloud");
var _region_map = require("./transforms/charts/region_map");
var _utils = require("./transforms/utils");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const compatibilityMap = {
  lnsMetric: 'metric',
  lnsLegacyMetric: 'legacy_metric',
  lnsXY: 'xy',
  lnsGauge: 'gauge',
  lnsHeatmap: 'heatmap',
  lnsTagcloud: 'tagcloud',
  lnsChoropleth: 'region_map'
};

/**
 * A minimal type to extend for type lookup
 */

const apiConvertersByChart = {
  metric: {
    fromAPItoLensState: _metric.fromAPItoLensState,
    fromLensStateToAPI: _metric.fromLensStateToAPI
  },
  legacy_metric: {
    fromAPItoLensState: _legacy_metric.fromAPItoLensState,
    fromLensStateToAPI: _legacy_metric.fromLensStateToAPI
  },
  xy: {
    fromAPItoLensState: _xy.fromAPItoLensState,
    fromLensStateToAPI: _xy.fromLensStateToAPI
  },
  gauge: {
    fromAPItoLensState: _gauge.fromAPItoLensState,
    fromLensStateToAPI: _gauge.fromLensStateToAPI
  },
  heatmap: {
    fromAPItoLensState: _heatmap.fromAPItoLensState,
    fromLensStateToAPI: _heatmap.fromLensStateToAPI
  },
  tagcloud: {
    fromAPItoLensState: _tagcloud.fromAPItoLensState,
    fromLensStateToAPI: _tagcloud.fromLensStateToAPI
  },
  region_map: {
    fromAPItoLensState: _region_map.fromAPItoLensState,
    fromLensStateToAPI: _region_map.fromLensStateToAPI
  }
};
class LensConfigBuilder {
  constructor(dataViewsAPI, enableAPITransforms = false) {
    (0, _defineProperty2.default)(this, "charts", {
      metric: _charts.buildMetric,
      tagcloud: _charts.buildTagCloud,
      treemap: _charts.buildPartitionChart,
      pie: _charts.buildPartitionChart,
      donut: _charts.buildPartitionChart,
      gauge: _charts.buildGauge,
      heatmap: _charts.buildHeatmap,
      mosaic: _charts.buildPartitionChart,
      regionmap: _charts.buildRegionMap,
      xy: _charts.buildXY,
      table: _charts.buildTable
    });
    (0, _defineProperty2.default)(this, "apiConvertersByChart", apiConvertersByChart);
    (0, _defineProperty2.default)(this, "dataViewsAPI", void 0);
    (0, _defineProperty2.default)(this, "enableAPITransforms", void 0);
    this.dataViewsAPI = dataViewsAPI;
    this.enableAPITransforms = enableAPITransforms;
  }
  get isEnabled() {
    return this.enableAPITransforms;
  }
  setEnabled(enabled) {
    this.enableAPITransforms = enabled;
  }
  isSupported(chartType) {
    var _compatibilityMap$cha;
    if (!this.enableAPITransforms) return false;
    if (!chartType) return false;
    const type = (_compatibilityMap$cha = compatibilityMap[chartType]) !== null && _compatibilityMap$cha !== void 0 ? _compatibilityMap$cha : chartType;
    return type in this.apiConvertersByChart;
  }
  getType(config) {
    if (config == null) {
      return null;
    }
    return 'visualizationType' in config ? config.visualizationType : (0, _utils2.isLensLegacyFormat)(config) ? config.chartType : 'type' in config ? config.type : null;
  }

  /**
   * Build a Lens configuration based on the provided API configuration
   * @param config ConfigBuilder API configuration
   * @param options
   * @returns Lens internal configuration
   */
  async build(config, options = {}) {
    if (!this.dataViewsAPI) {
      throw new Error('DataViews API is required to build Lens configurations');
    }
    const chartBuilderFn = this.charts[config.chartType];
    const chartConfig = await chartBuilderFn(config, {
      dataViewsAPI: this.dataViewsAPI
    });
    const chartState = {
      ...chartConfig,
      state: {
        ...chartConfig.state,
        filters: options.filters || [],
        query: options.query || {
          language: 'kuery',
          query: ''
        }
      }
    };
    if (options.embeddable) {
      return {
        id: (0, _uuid.v4)(),
        attributes: chartState,
        timeRange: options.timeRange,
        references: chartState.references
      };
    }
    return chartState;
  }
  fromAPIFormat(config) {
    const chartType = config.type;
    if (!(chartType in this.apiConvertersByChart)) {
      throw new Error(`No attributes converter found for chart type: ${chartType}`);
    }
    const converter = this.apiConvertersByChart[chartType];
    const attributes = converter.fromAPItoLensState(config); // handle type mismatches

    return {
      // @TODO investigate why it complains about missing type
      // type: 'lens',
      ...attributes,
      state: {
        ...attributes.state,
        query: {
          language: 'kuery',
          query: ''
        },
        ...(0, _utils.filtersAndQueryToLensState)(config)
      }
    };
  }
  toAPIFormat(config) {
    var _compatibilityMap$vis;
    const visType = config.visualizationType;
    const type = (_compatibilityMap$vis = compatibilityMap[visType]) !== null && _compatibilityMap$vis !== void 0 ? _compatibilityMap$vis : visType;
    if (!type || !(type in this.apiConvertersByChart)) {
      throw new Error(`No API converter found for chart type: ${visType} as ${type}`);
    }
    const converter = this.apiConvertersByChart[type];
    return {
      ...converter.fromLensStateToAPI(config),
      ...(0, _utils.filtersAndQueryToApiFormat)(config)
    };
  }
}
exports.LensConfigBuilder = LensConfigBuilder;