"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initMetricsSourceConfigurationRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _ioTsUtils = require("@kbn/io-ts-utils");
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/metrics-data-access-plugin/common");
var _get_has_data = require("../../../common/metrics_sources/get_has_data");
var _has_data = require("../../lib/sources/has_data");
var _create_search_client = require("../../lib/create_search_client");
var _errors = require("../../lib/sources/errors");
var _metrics_sources = require("../../../common/metrics_sources");
var _get_infra_metrics_client = require("../../lib/helpers/get_infra_metrics_client");
var _get_preferred_schema = require("../../lib/helpers/get_preferred_schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultStatus = {
  metricIndicesExist: false,
  remoteClustersExist: false
};
const initMetricsSourceConfigurationRoutes = libs => {
  const {
    framework,
    logger
  } = libs;
  const composeSourceStatus = async (requestContext, sourceId) => {
    try {
      const hasMetricIndices = await libs.sourceStatus.hasMetricIndices(requestContext, sourceId);
      return {
        metricIndicesExist: hasMetricIndices,
        remoteClustersExist: true
      };
    } catch (err) {
      logger.error(err);
      if (err instanceof _errors.NoSuchRemoteClusterError) {
        return defaultStatus;
      }
      return {
        metricIndicesExist: false,
        remoteClustersExist: true
      };
    }
  };
  framework.registerRoute({
    method: 'get',
    path: '/api/metrics/source/{sourceId}',
    validate: {
      params: _configSchema.schema.object({
        sourceId: _configSchema.schema.string()
      })
    }
  }, async (requestContext, request, response) => {
    const {
      sourceId
    } = request.params;
    const soClient = (await requestContext.core).savedObjects.client;
    try {
      const [sourceSettled, statusSettled] = await Promise.allSettled([libs.sources.getSourceConfiguration(soClient, sourceId), composeSourceStatus(requestContext, sourceId)]);
      const source = isFulfilled(sourceSettled) ? sourceSettled.value : null;
      const status = isFulfilled(statusSettled) ? statusSettled.value : defaultStatus;
      if (!source) {
        return response.notFound();
      }
      const sourceResponse = {
        source: {
          ...source,
          status
        }
      };
      return response.ok({
        body: _metrics_sources.metricsSourceConfigurationResponseRT.encode(sourceResponse)
      });
    } catch (error) {
      var _error$statusCode, _error$message;
      return response.customError({
        statusCode: (_error$statusCode = error.statusCode) !== null && _error$statusCode !== void 0 ? _error$statusCode : 500,
        body: {
          message: (_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'An unexpected error occurred'
        }
      });
    }
  });
  framework.registerRoute({
    method: 'patch',
    path: '/api/metrics/source/{sourceId}',
    validate: {
      params: _configSchema.schema.object({
        sourceId: _configSchema.schema.string()
      }),
      body: (0, _ioTsUtils.createRouteValidationFunction)(_metrics_sources.partialMetricsSourceConfigurationReqPayloadRT)
    }
  }, framework.router.handleLegacyErrors(async (requestContext, request, response) => {
    const {
      sources
    } = libs;
    const {
      sourceId
    } = request.params;
    const sourceConfigurationPayload = request.body;
    try {
      const soClient = (await requestContext.core).savedObjects.client;
      const sourceConfiguration = await sources.getSourceConfiguration(soClient, sourceId);
      if (sourceConfiguration.origin === 'internal') {
        response.conflict({
          body: 'A conflicting read-only source configuration already exists.'
        });
      }
      const sourceConfigurationExists = sourceConfiguration.origin === 'stored';
      const patchedSourceConfiguration = await (sourceConfigurationExists ? sources.updateSourceConfiguration(soClient, sourceId, sourceConfigurationPayload) : sources.createSourceConfiguration(soClient, sourceId, sourceConfigurationPayload));
      const status = await composeSourceStatus(requestContext, sourceId);
      const sourceResponse = {
        source: {
          ...patchedSourceConfiguration,
          status
        }
      };
      return response.ok({
        body: _metrics_sources.metricsSourceConfigurationResponseRT.encode(sourceResponse)
      });
    } catch (error) {
      var _error$statusCode2, _error$message2;
      if (_boom.default.isBoom(error)) {
        throw error;
      }
      if (error instanceof _errors.AnomalyThresholdRangeError) {
        return response.customError({
          statusCode: 400,
          body: {
            message: error.message
          }
        });
      }
      return response.customError({
        statusCode: (_error$statusCode2 = error.statusCode) !== null && _error$statusCode2 !== void 0 ? _error$statusCode2 : 500,
        body: {
          message: (_error$message2 = error.message) !== null && _error$message2 !== void 0 ? _error$message2 : 'An unexpected error occurred'
        }
      });
    }
  }));
  framework.registerRoute({
    method: 'get',
    path: '/api/metrics/source/{sourceId}/hasData',
    validate: {
      params: _configSchema.schema.object({
        sourceId: _configSchema.schema.string()
      })
    }
  }, async (requestContext, request, response) => {
    const {
      sourceId
    } = request.params;
    const client = (0, _create_search_client.createSearchClient)(requestContext, framework);
    const soClient = (await requestContext.core).savedObjects.client;
    const source = await libs.sources.getSourceConfiguration(soClient, sourceId);
    const results = await (0, _has_data.hasData)(source.configuration.metricAlias, client);
    return response.ok({
      body: {
        hasData: results,
        configuration: source.configuration
      }
    });
  });
  framework.registerRoute({
    method: 'get',
    path: '/api/metrics/source/hasData',
    validate: {
      query: (0, _ioTsUtils.createRouteValidationFunction)(_get_has_data.getHasDataQueryParamsRT)
    }
  }, async (context, request, response) => {
    try {
      const {
        source
      } = request.query;
      const infraMetricsClient = await (0, _get_infra_metrics_client.getInfraMetricsClient)({
        request,
        libs,
        context
      });
      const hostInventoryModel = (0, _common.findInventoryModel)('host');
      const hostIntegration = typeof (hostInventoryModel === null || hostInventoryModel === void 0 ? void 0 : hostInventoryModel.requiredIntegration) !== 'object' || !('otel' in (hostInventoryModel === null || hostInventoryModel === void 0 ? void 0 : hostInventoryModel.requiredIntegration)) ? undefined : hostInventoryModel.requiredIntegration;
      const hasDataResponse = await infraMetricsClient.search({
        track_total_hits: true,
        terminate_after: 1,
        size: 0,
        query: {
          bool: {
            should: source === 'all' ? [...(0, _server.existsQuery)(hostInventoryModel.fields.id), ...(0, _server.existsQuery)((0, _common.findInventoryFields)('container').id), ...(0, _server.existsQuery)((0, _common.findInventoryFields)('pod').id), ...(0, _server.existsQuery)((0, _common.findInventoryFields)('awsEC2').id), ...(0, _server.existsQuery)((0, _common.findInventoryFields)('awsS3').id), ...(0, _server.existsQuery)((0, _common.findInventoryFields)('awsRDS').id), ...(0, _server.existsQuery)((0, _common.findInventoryFields)('awsSQS').id)] : source === 'host' && hostIntegration ? [...(0, _server.termQuery)(_common.EVENT_MODULE, hostIntegration.beats), ...(0, _server.termQuery)(_common.METRICSET_MODULE, hostIntegration.beats), ...(0, _server.termQuery)(_common.DATASTREAM_DATASET, hostIntegration.otel)] : [],
            minimum_should_match: 1
          }
        }
      });
      return response.ok({
        body: _get_has_data.getHasDataResponseRT.encode({
          hasData: hasDataResponse.hits.total.value > 0
        })
      });
    } catch (err) {
      var _err$statusCode, _err$message;
      if (_boom.default.isBoom(err)) {
        return response.customError({
          statusCode: err.output.statusCode,
          body: {
            message: err.output.payload.message
          }
        });
      }
      return response.customError({
        statusCode: (_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 500,
        body: {
          message: (_err$message = err.message) !== null && _err$message !== void 0 ? _err$message : 'An unexpected error occurred'
        }
      });
    }
  });
  framework.registerRoute({
    method: 'get',
    path: '/api/metrics/source/time_range_metadata',
    validate: {
      query: (0, _ioTsUtils.createRouteValidationFunction)(_get_has_data.getTimeRangeMetadataQueryParamsRT)
    }
  }, async (context, request, response) => {
    try {
      const {
        from,
        to,
        dataSource,
        kuery,
        filters,
        isInventoryView
      } = request.query;
      const infraMetricsClient = await (0, _get_infra_metrics_client.getInfraMetricsClient)({
        request,
        libs,
        context
      });
      const {
        schemas,
        preferredSchema
      } = await (0, _get_preferred_schema.getPreferredSchema)({
        infraMetricsClient,
        dataSource,
        from,
        to,
        kuery,
        filters,
        isInventoryView
      });
      return response.ok({
        body: _get_has_data.getTimeRangeMetadataResponseRT.encode({
          schemas,
          preferredSchema
        })
      });
    } catch (err) {
      var _err$statusCode2, _err$message2;
      if (_boom.default.isBoom(err)) {
        return response.customError({
          statusCode: err.output.statusCode,
          body: {
            message: err.output.payload.message
          }
        });
      }
      return response.customError({
        statusCode: (_err$statusCode2 = err.statusCode) !== null && _err$statusCode2 !== void 0 ? _err$statusCode2 : 500,
        body: {
          message: (_err$message2 = err.message) !== null && _err$message2 !== void 0 ? _err$message2 : 'An unexpected error occurred'
        }
      });
    }
  });
};
exports.initMetricsSourceConfigurationRoutes = initMetricsSourceConfigurationRoutes;
const isFulfilled = promiseSettlement => promiseSettlement.status === 'fulfilled';