"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndicesMetadataService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _events = require("../ebt/events");
var _receiver = require("./receiver");
var _sender = require("./sender");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'IndicesMetadata:IndicesMetadataTask';
const TASK_ID = 'indices-metadata:indices-metadata-task:1.0.0';
const INTERVAL = '24h';
class IndicesMetadataService {
  constructor(logger, configurationService) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "configurationService", void 0);
    (0, _defineProperty2.default)(this, "receiver", void 0);
    (0, _defineProperty2.default)(this, "sender", void 0);
    (0, _defineProperty2.default)(this, "subscription$", void 0);
    (0, _defineProperty2.default)(this, "configuration", void 0);
    this.logger = logger.get(IndicesMetadataService.name);
    this.configurationService = configurationService;
  }
  setup(taskManager) {
    this.logger.debug('Setting up indices metadata service');
    this.registerIndicesMetadataTask(taskManager);
  }
  start(taskManager, analytics, esClient, isServerless) {
    this.logger.debug('Starting indices metadata service');
    this.receiver = new _receiver.MetadataReceiver(this.logger, esClient, isServerless);
    this.sender = new _sender.MetadataSender(this.logger, analytics);
    this.subscription$ = this.configurationService.getIndicesMetadataConfiguration$().subscribe(configuration => {
      this.logger.debug('Indices metadata configuration updated', {
        configuration
      });
      if (configuration) {
        this.configuration = configuration;
      }
    });
    this.scheduleIndicesMetadataTask(taskManager).catch(error => {
      this.logger.error('Failed to schedule Indices Metadata Task', {
        error
      });
    });
  }
  stop() {
    var _this$subscription$;
    this.logger.debug('Stopping indices metadata service');
    (_this$subscription$ = this.subscription$) === null || _this$subscription$ === void 0 ? void 0 : _this$subscription$.unsubscribe();
  }
  async publishIndicesMetadata() {
    this.ensureInitialized();
    if (!this.configuration || this.configuration.index_query_size === 0) {
      this.logger.debug('Index query size is 0, skipping indices metadata publish');
      return;
    }

    // 1. Get cluster stats and list of indices and datastreams
    const [indicesSettings, dataStreams, indexTemplates] = await Promise.all([this.receiver.getIndices(), this.receiver.getDataStreams(), this.receiver.getIndexTemplatesStats()]).catch(error => {
      this.logger.error('Error fetching indices metadata', {
        error
      });
      return [undefined, undefined, undefined];
    });
    if (indicesSettings === undefined || dataStreams === undefined || indexTemplates === undefined) {
      this.logger.debug('Skipping indices metadata publish due to fetch errors');
      return;
    }
    const indices = indicesSettings.map(index => index.index_name);

    // 2. Publish datastreams stats
    const dsCount = this.publishDatastreamsStats(dataStreams.slice(0, this.configuration.datastreams_threshold));

    // 3. Publish indices settings
    const indicesSettingsCount = this.publishIndicesSettings(indicesSettings.slice(0, this.configuration.indices_settings_threshold));

    // 4. Get and publish indices stats
    const indicesCount = await this.publishIndicesStats(indices.slice(0, this.configuration.indices_threshold));

    // 5. Get ILM stats and publish them
    let ilmNames = new Set();
    if (await this.receiver.isIlmStatsAvailable()) {
      ilmNames = await this.publishIlmStats(indices.slice(0, this.configuration.indices_threshold));
    } else {
      this.logger.debug('ILM explain API is not available');
    }

    // 6. Publish ILM policies
    const policyCount = await this.publishIlmPolicies(ilmNames);

    // 7. Publish index templates
    const indexTemplatesCount = await this.publishIndexTemplatesStats(indexTemplates.slice(0, this.configuration.indices_threshold));
    this.logger.debug('EBT events sent', {
      datastreams: dsCount,
      ilms: ilmNames.size,
      indices: indicesCount,
      indicesSettings: indicesSettingsCount,
      policies: policyCount,
      templates: indexTemplatesCount
    });
  }
  ensureInitialized() {
    if (!this.receiver || !this.sender) {
      throw new Error('Indices metadata service not initialized');
    }
  }
  publishDatastreamsStats(stats) {
    const events = {
      items: stats
    };
    this.sender.reportEBT(_events.DATA_STREAM_EVENT, events);
    this.logger.debug('Data streams events sent', {
      count: events.items.length
    });
    return events.items.length;
  }
  registerIndicesMetadataTask(taskManager) {
    const service = this;
    this.logger.debug('About to register task', {
      task: TASK_ID
    });
    taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'Metrics Data Access - Indices Metadata Task',
        description: 'This task periodically pushes indices metadata to the telemetry service.',
        timeout: '2m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            async run() {
              const {
                state
              } = taskInstance;
              await service.publishIndicesMetadata();
              return {
                state
              };
            },
            async cancel() {
              service.logger.warn('Task timed out', {
                task: TASK_ID
              });
            }
          };
        }
      }
    });
    this.logger.debug('Task registered', {
      task: TASK_ID,
      type: TASK_TYPE
    });
  }
  async scheduleIndicesMetadataTask(taskManager) {
    var _taskInstance$schedul;
    this.logger.debug('About to schedule task', {
      task: TASK_ID
    });
    const taskInstance = await taskManager.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      schedule: {
        interval: INTERVAL
      },
      params: {},
      state: {},
      scope: ['uptime']
    });
    this.logger.debug('Task scheduled', {
      task: TASK_ID,
      interval: (_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval
    });
    return taskInstance;
  }
  async publishIndicesStats(indices) {
    if (!this.configuration) {
      return 0;
    }
    const indicesStats = {
      items: []
    };
    for await (const stat of this.receiver.getIndicesStats(indices, this.configuration.index_query_size)) {
      indicesStats.items.push(stat);
    }
    this.sender.reportEBT(_events.INDEX_STATS_EVENT, indicesStats);
    this.logger.debug('Indices stats sent', {
      count: indicesStats.items.length
    });
    return indicesStats.items.length;
  }
  publishIndicesSettings(settings) {
    const indicesSettings = {
      items: settings
    };
    this.sender.reportEBT(_events.INDEX_SETTINGS_EVENT, indicesSettings);
    this.logger.debug('Indices settings sent', {
      count: indicesSettings.items.length
    });
    return indicesSettings.items.length;
  }
  async publishIlmStats(indices) {
    const ilmNames = new Set();
    const ilmsStats = {
      items: []
    };
    for await (const stat of this.receiver.getIlmsStats(indices)) {
      if (stat.policy_name !== undefined) {
        ilmNames.add(stat.policy_name);
        ilmsStats.items.push(stat);
      }
    }
    this.sender.reportEBT(_events.ILM_STATS_EVENT, ilmsStats);
    this.logger.debug('ILM stats sent', {
      count: ilmNames.size
    });
    return ilmNames;
  }
  async publishIlmPolicies(ilmNames) {
    if (!this.configuration) {
      return 0;
    }
    const ilmPolicies = {
      items: []
    };
    for await (const policy of this.receiver.getIlmsPolicies(Array.from(ilmNames.values()), this.configuration.ilm_policy_query_size)) {
      ilmPolicies.items.push(policy);
    }
    this.sender.reportEBT(_events.ILM_POLICY_EVENT, ilmPolicies);
    this.logger.debug('ILM policies sent', {
      count: ilmPolicies.items.length
    });
    return ilmPolicies.items.length;
  }
  async publishIndexTemplatesStats(indexTemplates) {
    const templateStats = {
      items: indexTemplates
    };
    this.sender.reportEBT(_events.INDEX_TEMPLATES_EVENT, templateStats);
    this.logger.debug('Index templates stats sent', {
      count: templateStats.items.length
    });
    return templateStats.items.length;
  }
}
exports.IndicesMetadataService = IndicesMetadataService;