"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateAndSanitizeFileData = validateAndSanitizeFileData;
var _svg = require("../sanitizations/svg");
var _file_size = require("./file_size");
var _file_mimetype = require("./file_mimetype");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateAndSanitizeFileData(data, filePath) {
  const dataBuffer = toBuffer(data);

  // If data cannot be converted to a Buffer (e.g. AsyncIterable or Stream), return the original data
  if (!dataBuffer) {
    return data;
  }
  (0, _file_size.validateFileSize)(dataBuffer);
  const possibleMimeTypes = (0, _file_mimetype.validateMimeType)(dataBuffer, filePath);
  if (possibleMimeTypes.includes('image/svg+xml')) {
    try {
      return (0, _svg.sanitizeSvg)(dataBuffer);
    } catch (error) {
      throw new Error(`Failed to sanitize SVG content: ${error.message}`);
    }
  }
  return dataBuffer;
}
function isObjectWithBufferProperty(data) {
  return data !== null && typeof data === 'object' && 'buffer' in data && Buffer.isBuffer(data.buffer);
}
function isIterable(data) {
  return data !== null && typeof data === 'object' && Symbol.iterator in data;
}
function toBuffer(data) {
  if (Buffer.isBuffer(data)) {
    return data;
  }
  if (ArrayBuffer.isView(data)) {
    return Buffer.from(data.buffer, data.byteOffset, data.byteLength);
  }
  if (typeof data === 'string') {
    return Buffer.from(data, 'utf8');
  }
  if (isObjectWithBufferProperty(data)) {
    return data.buffer;
  }
  if (isIterable(data)) {
    const chunks = [];
    for (const chunk of data) {
      if (Buffer.isBuffer(chunk)) {
        chunks.push(chunk);
      } else if (ArrayBuffer.isView(chunk)) {
        chunks.push(Buffer.from(chunk.buffer, chunk.byteOffset, chunk.byteLength));
      } else if (typeof chunk === 'string') {
        chunks.push(Buffer.from(chunk, 'utf8'));
      } else {
        throw new Error('Unsupported chunk type in Iterable');
      }
    }
    return Buffer.concat(chunks);
  }
  return null;
}