"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runReinstallPackagesForGlobalAssetUpdate = runReinstallPackagesForGlobalAssetUpdate;
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _pMap = _interopRequireDefault(require("p-map"));
var _services = require("../../services");
var _constants = require("../../constants");
var _packages = require("../../services/epm/packages");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Reinstalls all installed packages after global Fleet ES assets have been updated.
 * This ensures packages reference the latest global component templates and ingest pipelines.
 *
 * This task is scheduled when global assets (component templates, ingest pipelines) are
 * created or updated during Fleet setup, typically during a stack upgrade.
 */
async function runReinstallPackagesForGlobalAssetUpdate({
  abortController,
  logger
}) {
  var _config$startupOptimi, _config$startupOptimi2;
  const soClient = _services.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const esClient = _services.appContextService.getInternalUserESClient();
  const config = _services.appContextService.getConfig();
  const maxConcurrency = (_config$startupOptimi = config === null || config === void 0 ? void 0 : (_config$startupOptimi2 = config.startupOptimization) === null || _config$startupOptimi2 === void 0 ? void 0 : _config$startupOptimi2.maxConcurrentPackageOperations) !== null && _config$startupOptimi !== void 0 ? _config$startupOptimi : _constants.MAX_CONCURRENT_EPM_PACKAGES_INSTALLATIONS;
  logger.info('Starting package reinstallation after global Fleet asset update');
  const installedPackages = await (0, _packages.getInstallations)(soClient);
  const installations = installedPackages.saved_objects.map(so => so.attributes);
  if (installations.length === 0) {
    logger.debug('No packages installed, skipping reinstallation');
    return;
  }
  logger.info(`Reinstalling ${installations.length} packages after global asset changes`);
  let successCount = 0;
  let errorCount = 0;
  await (0, _pMap.default)(installations, async installation => {
    if (abortController.signal.aborted) {
      return;
    }
    try {
      await (0, _packages.reinstallPackageForInstallation)({
        soClient,
        esClient,
        installation
      });
      successCount++;
      logger.debug(`Successfully reinstalled package ${installation.name}`);
    } catch (err) {
      errorCount++;
      _elasticApmNode.default.captureError(err);
      logger.error(`Package needs to be manually reinstalled ${installation.name} after installing Fleet global assets: ${err.message}`);
    }
  }, {
    concurrency: maxConcurrency
  });
  if (abortController.signal.aborted) {
    logger.warn('Package reinstallation was aborted');
    return;
  }
  logger.info(`Completed package reinstallation after global asset update: ${successCount} succeeded, ${errorCount} failed`);
}