"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TYPE = exports.HAS_CHANGED_RUNTIME_FIELD = exports.AgentStatusChangeTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _server = require("@kbn/core/server");
var _task = require("@kbn/task-manager-plugin/server/task");
var _elasticsearch = require("@elastic/elasticsearch");
var _agent = require("../../common/constants/agent");
var _services = require("../services");
var _agents = require("../services/agents");
var _constants = require("../constants");
var _agent_policy = require("../services/agent_policy");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'fleet:agent-status-change-task';
const VERSION = exports.VERSION = '1.0.2';
const TITLE = 'Fleet Agent Status Change Task';
const SCOPE = ['fleet'];
const DEFAULT_INTERVAL = '1m';
const TIMEOUT = '1m';
const AGENTS_BATCHSIZE = 10000;
const HAS_CHANGED_RUNTIME_FIELD = exports.HAS_CHANGED_RUNTIME_FIELD = {
  hasChanged: {
    type: 'boolean',
    script: {
      lang: 'painless',
      source: "emit(doc['last_known_status'].size() == 0 || doc['status'].size() == 0 || doc['last_known_status'].value != doc['status'].value );"
    }
  }
};
class AgentStatusChangeTask {
  constructor(setupContract) {
    var _config$taskInterval;
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "taskInterval", void 0);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error('[AgentStatusChangeTask] Missing required service during start');
        return;
      }
      this.wasStarted = true;
      this.logger.info(`[AgentStatusChangeTask] Started with interval of [${this.taskInterval}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: TYPE,
          scope: SCOPE,
          schedule: {
            interval: this.taskInterval
          },
          state: {},
          params: {
            version: VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task AgentStatusChangeTask, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core, abortController) => {
      if (!_services.appContextService.getExperimentalFeatures().enableAgentStatusAlerting) {
        this.logger.debug('[AgentStatusChangeTask] Aborting runTask: agent status alerting feature is disabled');
        return;
      }
      if (!this.wasStarted) {
        this.logger.debug('[AgentStatusChangeTask] runTask Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        this.logger.debug(`[AgentStatusChangeTask] Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      this.logger.debug(`[runTask()] started`);
      const [coreStart, _startDeps] = await core.getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      try {
        await this.persistAgentStatusChanges(esClient, soClient, abortController);
        this.endRun('success');
      } catch (err) {
        if (err instanceof _elasticsearch.errors.RequestAbortedError) {
          this.logger.warn(`[AgentStatusChangeTask] request aborted due to timeout: ${err}`);
          this.endRun();
          return;
        }
        this.logger.error(`[AgentStatusChangeTask] error: ${err}`);
        this.endRun('error');
      }
    });
    (0, _defineProperty2.default)(this, "persistAgentStatusChanges", async (esClient, soClient, abortController) => {
      let agentlessPolicies;
      const agentsFetcher = await (0, _agents.fetchAllAgentsByKuery)(esClient, soClient, {
        perPage: AGENTS_BATCHSIZE,
        kuery: 'hasChanged:true',
        runtimeFields: HAS_CHANGED_RUNTIME_FIELD
      });
      for await (const agentPageResults of agentsFetcher) {
        if (!agentPageResults.length) {
          this.endRun('Found no agents to process');
          return;
        }
        const updateErrors = {};
        const agentsToUpdate = agentPageResults;
        (0, _utils.throwIfAborted)(abortController);
        if (agentsToUpdate.length === 0) {
          continue;
        } else {
          this.logger.debug(`[AgentStatusChangeTask] Recording ${agentsToUpdate.length} status changes`);
        }
        if (!agentlessPolicies) {
          agentlessPolicies = await this.findAgentlessPolicies();
        }
        await this.bulkCreateAgentStatusChangeDocs(esClient, agentsToUpdate, agentlessPolicies);
        await (0, _agents.bulkUpdateAgents)(esClient, agentsToUpdate.map(agent => ({
          agentId: agent.id,
          data: {
            last_known_status: agent.status
          }
        })), updateErrors);
        if (Object.keys(updateErrors).length > 0) {
          this.logger.info(`Errors while bulk updating agents: ${JSON.stringify(updateErrors)}`);
        }
      }
    });
    (0, _defineProperty2.default)(this, "findAgentlessPolicies", async () => {
      const internalSoClientWithoutSpaceExtension = _services.appContextService.getInternalUserSOClientWithoutSpaceExtension();
      const agentlessPolicies = await _services.agentPolicyService.list(internalSoClientWithoutSpaceExtension, {
        spaceId: '*',
        perPage: _constants.SO_SEARCH_LIMIT,
        kuery: `${await (0, _agent_policy.getAgentPolicySavedObjectType)()}.supports_agentless:true`,
        fields: ['id']
      });
      return agentlessPolicies.items.map(policy => policy.id);
    });
    (0, _defineProperty2.default)(this, "bulkCreateAgentStatusChangeDocs", async (esClient, agentsToUpdate, agentlessPolicies) => {
      const bulkBody = agentsToUpdate.flatMap(agent => {
        var _ref;
        const body = {
          '@timestamp': new Date().toISOString(),
          data_stream: _agent.AGENT_STATUS_CHANGE_DATA_STREAM,
          agent: {
            id: agent.id
          },
          status: agent.status,
          policy_id: agent.policy_id,
          space_id: agent.namespaces,
          hostname: agent.local_metadata.host.hostname,
          agentless: (_ref = agent.policy_id && (agentlessPolicies === null || agentlessPolicies === void 0 ? void 0 : agentlessPolicies.includes(agent.policy_id))) !== null && _ref !== void 0 ? _ref : false
        };
        return [{
          create: {
            _id: (0, _uuid.v4)()
          }
        }, body];
      });
      await esClient.bulk({
        index: _agent.AGENT_STATUS_CHANGE_DATA_STREAM_NAME,
        operations: bulkBody,
        refresh: 'wait_for'
      });
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory,
      config
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    this.taskInterval = (_config$taskInterval = config.taskInterval) !== null && _config$taskInterval !== void 0 ? _config$taskInterval : DEFAULT_INTERVAL;
    _taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance,
          abortController
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, _core, abortController);
            },
            cancel: async () => {}
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  endRun(msg = '') {
    this.logger.debug(`[AgentStatusChangeTask] runTask ended${msg ? ': ' + msg : ''}`);
  }
}
exports.AgentStatusChangeTask = AgentStatusChangeTask;