"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._runSetupWithLock = _runSetupWithLock;
exports.ensureFleetDirectories = ensureFleetDirectories;
exports.formatNonFatalErrors = formatNonFatalErrors;
exports.setupFleet = setupFleet;
var _promises = _interopRequireDefault(require("fs/promises"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _lodash = require("lodash");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _lockManager = require("@kbn/lock-manager");
var _errors = require("../../common/errors");
var _constants2 = require("../../common/constants");
var _schedule = require("../tasks/setup/schedule");
var _app_context = require("./app_context");
var _preconfiguration = require("./preconfiguration");
var _outputs = require("./preconfiguration/outputs");
var _fleet_proxies = require("./preconfiguration/fleet_proxies");
var _output = require("./output");
var _download_source = require("./download_source");
var _ = require(".");
var _setup_utils = require("./setup_utils");
var _install = require("./epm/packages/install");
var _managed_package_policies = require("./setup/managed_package_policies");
var _upgrade_package_install_version = require("./setup/upgrade_package_install_version");
var _upgrade_agent_policy_schema_version = require("./setup/upgrade_agent_policy_schema_version");
var _fleet_server_host = require("./fleet_server_host");
var _fleet_server_host2 = require("./preconfiguration/fleet_server_host");
var _clean_old_fleet_indices = require("./setup/clean_old_fleet_indices");
var _fleet_server_policies_enrollment_keys = require("./setup/fleet_server_policies_enrollment_keys");
var _space_settings = require("./preconfiguration/space_settings");
var _delete_unenrolled_agent_setting = require("./preconfiguration/delete_unenrolled_agent_setting");
var _backfill_agentless = require("./backfill_agentless");
var _update_deprecated_component_templates = require("./setup/update_deprecated_component_templates");
var _fleet_synced_integrations = require("./setup/fleet_synced_integrations");
var _agentless_settings_ids = require("./agentless_settings_ids");
var _saved_objects = require("./epm/kibana/assets/saved_objects");
var _ensure_fleet_global_es_assets = require("./setup/ensure_fleet_global_es_assets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function _runSetupWithLock(setupFn) {
  return await (0, _pRetry.default)(() => _app_context.appContextService.getLockManagerService().withLock('fleet-setup', () => setupFn()), {
    onFailedAttempt: async error => {
      if (!(error instanceof _lockManager.LockAcquisitionError)) {
        throw error;
      }
    },
    maxRetryTime: 5 * 60 * 1000 // Retry for 5 minute to get the lock
  });
}
async function setupFleet(soClient, esClient, options = {
  useLock: false
}) {
  const t = _elasticApmNode.default.startTransaction('fleet-setup', 'fleet');
  try {
    if (options.useLock) {
      return _runSetupWithLock(() => (0, _setup_utils.awaitIfPending)(async () => createSetupSideEffects(soClient, esClient)));
    } else {
      return await (0, _setup_utils.awaitIfPending)(async () => createSetupSideEffects(soClient, esClient));
    }
  } catch (error) {
    _elasticApmNode.default.captureError(error);
    t.setOutcome('failure');
    throw error;
  } finally {
    t.end();
  }
}
async function createSetupSideEffects(soClient, esClient) {
  var _appContextService$ge, _appContextService$ge2, _appContextService$ge3, _stepSpan, _stepSpan2, _stepSpan3, _config$startupOptimi, _config$startupOptimi2, _stepSpan4, _appContextService$ge4, _stepSpan5, _stepSpan6, _stepSpan7;
  const logger = _app_context.appContextService.getLogger();
  logger.info('Beginning fleet setup');
  await (0, _clean_old_fleet_indices.cleanUpOldFileIndices)(esClient, logger);
  await ensureFleetDirectories();
  const {
    agentPolicies: policiesOrUndefined,
    packages: packagesOrUndefined
  } = (_appContextService$ge = _app_context.appContextService.getConfig()) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : {};
  const policies = policiesOrUndefined !== null && policiesOrUndefined !== void 0 ? policiesOrUndefined : [];
  let packages = packagesOrUndefined !== null && packagesOrUndefined !== void 0 ? packagesOrUndefined : [];
  logger.debug('Setting Fleet server config');
  await (0, _fleet_server_host.migrateSettingsToFleetServerHost)(soClient, esClient);
  logger.debug('Setting up Fleet download source');
  const defaultDownloadSource = await _download_source.downloadSourceService.ensureDefault(soClient, esClient);
  // Need to be done before outputs and fleet server hosts as these object can reference a proxy
  logger.debug('Setting up Proxy');
  await (0, _fleet_proxies.ensurePreconfiguredFleetProxies)(soClient, esClient, (0, _fleet_proxies.getPreconfiguredFleetProxiesFromConfig)(_app_context.appContextService.getConfig()));
  logger.debug('Setting up Fleet Sever Hosts');
  await (0, _fleet_server_host2.ensurePreconfiguredFleetServerHosts)(soClient, esClient, (0, _fleet_server_host2.getPreconfiguredFleetServerHostFromConfig)(_app_context.appContextService.getConfig()));
  logger.debug('Setting up Space settings');
  await (0, _space_settings.ensureSpaceSettings)((_appContextService$ge2 = (_appContextService$ge3 = _app_context.appContextService.getConfig()) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.spaceSettings) !== null && _appContextService$ge2 !== void 0 ? _appContextService$ge2 : []);
  logger.debug('Setting up delete unenrolled agents setting');
  await (0, _delete_unenrolled_agent_setting.ensureDeleteUnenrolledAgentsSetting)(soClient, (0, _delete_unenrolled_agent_setting.getPreconfiguredDeleteUnenrolledAgentsSettingFromConfig)(_app_context.appContextService.getConfig()));
  logger.debug('Setting up Fleet outputs');
  await _.settingsService.settingsSetup(soClient);
  await (0, _outputs.ensurePreconfiguredOutputs)(soClient, esClient, (0, _outputs.getPreconfiguredOutputFromConfig)(_app_context.appContextService.getConfig()));
  const defaultOutput = await _output.outputService.ensureDefaultOutput(soClient, esClient);
  logger.debug('Backfilling output performance presets');
  await _output.outputService.backfillAllOutputPresets(soClient, esClient);
  logger.debug('Setting up Fleet Elasticsearch assets');
  let stepSpan = _elasticApmNode.default.startSpan('Install Fleet global assets', 'preconfiguration');
  await (0, _ensure_fleet_global_es_assets.ensureFleetGlobalEsAssets)({
    soClient,
    esClient,
    logger
  }, {
    reinstallPackages: true
  });
  (_stepSpan = stepSpan) === null || _stepSpan === void 0 ? void 0 : _stepSpan.end();

  // Ensure that required packages are always installed even if they're left out of the config
  const preconfiguredPackageNames = new Set(packages.map(pkg => pkg.name));
  const autoUpdateablePackages = (0, _lodash.compact)(await Promise.all(_constants2.AUTO_UPDATE_PACKAGES.map(pkg => (0, _install.isPackageInstalled)({
    savedObjectsClient: soClient,
    pkgName: pkg.name
  }).then(installed => installed ? pkg : undefined))));
  packages = [...packages, ...autoUpdateablePackages.filter(pkg => !preconfiguredPackageNames.has(pkg.name))];
  logger.debug('Setting up initial Fleet packages');
  stepSpan = _elasticApmNode.default.startSpan('Install preconfigured packages and policies', 'preconfiguration');
  const {
    nonFatalErrors: preconfiguredPackagesNonFatalErrors
  } = await (0, _preconfiguration.ensurePreconfiguredPackagesAndPolicies)(soClient, esClient, policies, packages, defaultOutput, defaultDownloadSource, _constants.DEFAULT_SPACE_ID);
  (_stepSpan2 = stepSpan) === null || _stepSpan2 === void 0 ? void 0 : _stepSpan2.end();
  stepSpan = _elasticApmNode.default.startSpan('Upgrade managed package policies', 'preconfiguration');
  await (0, _managed_package_policies.setupUpgradeManagedPackagePolicies)(soClient, esClient);
  (_stepSpan3 = stepSpan) === null || _stepSpan3 === void 0 ? void 0 : _stepSpan3.end();
  logger.debug('Upgrade Fleet package install versions');
  stepSpan = _elasticApmNode.default.startSpan('Upgrade package install format version', 'preconfiguration');
  const config = _app_context.appContextService.getConfig();
  const shouldDeferPackageBumpInstallVersion = (_config$startupOptimi = config === null || config === void 0 ? void 0 : (_config$startupOptimi2 = config.startupOptimization) === null || _config$startupOptimi2 === void 0 ? void 0 : _config$startupOptimi2.deferPackageBumpInstallVersion) !== null && _config$startupOptimi !== void 0 ? _config$startupOptimi : false;
  if (shouldDeferPackageBumpInstallVersion) {
    logger.info('Deferring package install version upgrade to background task');
    await (0, _schedule.scheduleSetupTask)(_app_context.appContextService.getTaskManagerStart(), {
      type: 'upgradePackageInstallVersion'
    });
  } else {
    await (0, _upgrade_package_install_version.upgradePackageInstallVersion)({
      soClient,
      esClient,
      logger
    });
  }
  (_stepSpan4 = stepSpan) === null || _stepSpan4 === void 0 ? void 0 : _stepSpan4.end();
  logger.debug('Generating key pair for message signing');
  stepSpan = _elasticApmNode.default.startSpan('Configure message signing', 'preconfiguration');
  if (!((_appContextService$ge4 = _app_context.appContextService.getMessageSigningService()) !== null && _appContextService$ge4 !== void 0 && _appContextService$ge4.isEncryptionAvailable)) {
    logger.warn('xpack.encryptedSavedObjects.encryptionKey is not configured, private key passphrase is being stored in plain text');
  }
  let messageSigningServiceNonFatalError;
  try {
    var _appContextService$ge5;
    await ((_appContextService$ge5 = _app_context.appContextService.getMessageSigningService()) === null || _appContextService$ge5 === void 0 ? void 0 : _appContextService$ge5.generateKeyPair());
  } catch (error) {
    if (error instanceof _errors.MessageSigningError) {
      messageSigningServiceNonFatalError = {
        error
      };
    } else {
      throw error;
    }
  }
  (_stepSpan5 = stepSpan) === null || _stepSpan5 === void 0 ? void 0 : _stepSpan5.end();
  stepSpan = _elasticApmNode.default.startSpan('Upgrade agent policy schema', 'preconfiguration');
  logger.debug('Upgrade Agent policy schema version');
  await (0, _upgrade_agent_policy_schema_version.upgradeAgentPolicySchemaVersion)(soClient);
  (_stepSpan6 = stepSpan) === null || _stepSpan6 === void 0 ? void 0 : _stepSpan6.end();
  stepSpan = _elasticApmNode.default.startSpan('Set up enrollment keys for preconfigured policies', 'preconfiguration');
  logger.debug('Setting up Fleet enrollment keys and verifying fleet server policies are not out-of-sync');
  await (0, _fleet_server_policies_enrollment_keys.ensureAgentPoliciesFleetServerKeysAndPolicies)({
    soClient,
    esClient,
    logger
  });
  (_stepSpan7 = stepSpan) === null || _stepSpan7 === void 0 ? void 0 : _stepSpan7.end();
  let backfillPackagePolicySupportsAgentlessError;
  try {
    logger.debug('Backfilling package policy supports_agentless field');
    await (0, _backfill_agentless.backfillPackagePolicySupportsAgentless)(esClient);
  } catch (error) {
    backfillPackagePolicySupportsAgentlessError = {
      error
    };
  }
  let ensureCorrectAgentlessSettingsIdsError;
  try {
    logger.debug('Fix agentless policy settings');
    await (0, _agentless_settings_ids.ensureCorrectAgentlessSettingsIds)(esClient);
  } catch (error) {
    ensureCorrectAgentlessSettingsIdsError = {
      error
    };
  }
  logger.debug('Update deprecated _source.mode in component templates');
  await (0, _update_deprecated_component_templates.updateDeprecatedComponentTemplates)(esClient);
  logger.debug('Create CCS index patterns for remote clusters');
  const {
    savedObjectsImporter
  } = (0, _saved_objects.getSpaceAwareSaveobjectsClients)();
  await (0, _fleet_synced_integrations.createCCSIndexPatterns)(esClient, soClient, savedObjectsImporter);
  const nonFatalErrors = [...preconfiguredPackagesNonFatalErrors, ...(messageSigningServiceNonFatalError ? [messageSigningServiceNonFatalError] : []), ...(backfillPackagePolicySupportsAgentlessError ? [backfillPackagePolicySupportsAgentlessError] : []), ...(ensureCorrectAgentlessSettingsIdsError ? [ensureCorrectAgentlessSettingsIdsError] : [])];
  logger.info('Scheduling async setup tasks');
  await (0, _schedule.scheduleSetupTask)(_app_context.appContextService.getTaskManagerStart());
  if (nonFatalErrors.length > 0) {
    logger.info('Encountered non fatal errors during Fleet setup');
    formatNonFatalErrors(nonFatalErrors).map(e => JSON.stringify(e)).forEach(error => {
      logger.info(error);
      _elasticApmNode.default.captureError(error);
    });
  }
  logger.info('Fleet setup completed');
  return {
    isInitialized: true,
    nonFatalErrors
  };
}

/**
 * Maps the `nonFatalErrors` object returned by the setup process to a more readable
 * and predictable format suitable for logging output or UI presentation.
 */
function formatNonFatalErrors(nonFatalErrors) {
  return nonFatalErrors.flatMap(e => {
    if ('error' in e) {
      return {
        name: e.error.name,
        message: e.error.message
      };
    } else if ('errors' in e) {
      return e.errors.map(upgradePackagePolicyError => {
        if (typeof upgradePackagePolicyError === 'string') {
          return {
            name: 'SetupNonFatalError',
            message: upgradePackagePolicyError
          };
        }
        return {
          name: upgradePackagePolicyError.key,
          message: upgradePackagePolicyError.message
        };
      });
    }
  });
}

/**
 * Confirm existence of various directories used by Fleet and warn if they don't exist
 */
async function ensureFleetDirectories() {
  var _config$developer;
  const logger = _app_context.appContextService.getLogger();
  const config = _app_context.appContextService.getConfig();
  const bundledPackageLocation = config === null || config === void 0 ? void 0 : (_config$developer = config.developer) === null || _config$developer === void 0 ? void 0 : _config$developer.bundledPackageLocation;
  const registryUrl = (0, _.getRegistryUrl)();
  if (!bundledPackageLocation) {
    logger.warn('xpack.fleet.developer.bundledPackageLocation is not configured');
    return;
  }
  try {
    await _promises.default.stat(bundledPackageLocation);
    logger.debug(`Bundled package directory ${bundledPackageLocation} exists`);
  } catch (error) {
    logger.warn(`Bundled package directory ${bundledPackageLocation} does not exist. All packages will be sourced from ${registryUrl}.`);
  }
}