"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractAndCreateCloudConnectorSecrets = extractAndCreateCloudConnectorSecrets;
var _errors = require("../../errors");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Extracts cloud connector variables from a package policy's inputs and creates secrets for non-secret-ref values
 * This function handles cloud connector secret creation separately from package policy secrets,
 * decoupling cloud connector secret handling from extractAndWriteSecrets.
 *
 * @param cloudProvider - The cloud provider (aws, azure, gcp)
 * @param packagePolicy - The package policy containing cloud connector vars
 * @param esClient - Elasticsearch client for creating secrets
 * @param logger - Logger instance
 * @returns CloudConnectorVars with secret references populated
 */
async function extractAndCreateCloudConnectorSecrets(cloudProvider, packagePolicy, esClient, logger) {
  logger.debug('Extracting package policy vars for cloud connector and creating secrets');
  if (packagePolicy.supports_cloud_connector && cloudProvider === 'aws') {
    return await extractAwsCloudConnectorSecrets(packagePolicy, esClient, logger);
  }
  if (packagePolicy.supports_cloud_connector && cloudProvider === 'azure') {
    return await extractAzureCloudConnectorSecrets(packagePolicy, esClient, logger);
  }
}

/**
 * Extracts AWS cloud connector variables and creates secrets
 */
async function extractAwsCloudConnectorSecrets(packagePolicy, esClient, logger) {
  var _packagePolicy$inputs, _packagePolicy$inputs2, _vars$role_arn, _vars$awsRole_arn;
  const vars = (_packagePolicy$inputs = packagePolicy.inputs.find(input => input.enabled)) === null || _packagePolicy$inputs === void 0 ? void 0 : (_packagePolicy$inputs2 = _packagePolicy$inputs.streams[0]) === null || _packagePolicy$inputs2 === void 0 ? void 0 : _packagePolicy$inputs2.vars;
  if (!vars) {
    logger.error('Package policy must contain vars for AWS cloud connector');
    throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain vars');
  }

  // Look for role_arn and external_id in the vars
  const roleArn = ((_vars$role_arn = vars.role_arn) === null || _vars$role_arn === void 0 ? void 0 : _vars$role_arn.value) || ((_vars$awsRole_arn = vars['aws.role_arn']) === null || _vars$awsRole_arn === void 0 ? void 0 : _vars$awsRole_arn.value);
  const externalIdVar = vars.external_id || vars['aws.credentials.external_id'];
  if (roleArn && externalIdVar) {
    let externalIdWithSecretRef;

    // If external_id is not already a secret reference, create a secret for it
    if (externalIdVar.value && !externalIdVar.value.isSecretRef) {
      logger.debug('Creating secret for AWS external_id');
      const secrets = await (0, _common.createSecrets)({
        esClient,
        values: [externalIdVar.value]
      });
      const firstSecret = secrets[0];
      if (Array.isArray(firstSecret)) {
        throw new _errors.CloudConnectorInvalidVarsError('Unexpected array of secrets for external_id');
      }
      externalIdWithSecretRef = {
        type: 'password',
        value: {
          id: firstSecret.id,
          isSecretRef: true
        }
      };
    } else {
      // Already a secret reference, ensure it has the correct type
      externalIdWithSecretRef = {
        type: 'password',
        value: externalIdVar.value
      };
    }
    const awsCloudConnectorVars = {
      role_arn: {
        type: 'text',
        value: roleArn
      },
      external_id: externalIdWithSecretRef
    };
    logger.debug(`Extracted AWS cloud connector vars: role_arn=${!!roleArn}, external_id=${!!externalIdWithSecretRef}`);
    return awsCloudConnectorVars;
  }
  logger.error('AWS cloud connector vars not found or incomplete');
  throw new _errors.CloudConnectorInvalidVarsError('Missing required AWS cloud connector variables: role_arn and external_id');
}

/**
 * Extracts Azure cloud connector variables and creates secrets
 */
async function extractAzureCloudConnectorSecrets(packagePolicy, esClient, logger) {
  var _packagePolicy$inputs3, _packagePolicy$inputs4;
  const vars = (_packagePolicy$inputs3 = packagePolicy.inputs.find(input => input.enabled)) === null || _packagePolicy$inputs3 === void 0 ? void 0 : (_packagePolicy$inputs4 = _packagePolicy$inputs3.streams[0]) === null || _packagePolicy$inputs4 === void 0 ? void 0 : _packagePolicy$inputs4.vars;
  if (!vars) {
    logger.error('Package policy must contain vars for Azure cloud connector');
    throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain vars');
  }
  const tenantIdVar = vars.tenant_id || vars['azure.credentials.tenant_id'];
  const clientIdVar = vars.client_id || vars['azure.credentials.client_id'];
  const azureCredentials = vars.azure_credentials_cloud_connector_id || vars['azure.credentials.cloud_connector_id'];
  if (tenantIdVar && clientIdVar && azureCredentials) {
    let tenantIdWithSecretRef = tenantIdVar;
    let clientIdWithSecretRef = clientIdVar;

    // Create secrets for tenant_id and client_id if they're not already secret references
    const secretsToCreate = [];
    if (tenantIdVar.value && !tenantIdVar.value.isSecretRef) {
      secretsToCreate.push(tenantIdVar.value);
    }
    if (clientIdVar.value && !clientIdVar.value.isSecretRef) {
      secretsToCreate.push(clientIdVar.value);
    }
    if (secretsToCreate.length > 0) {
      logger.debug(`Creating ${secretsToCreate.length} secrets for Azure cloud connector`);
      const secrets = await (0, _common.createSecrets)({
        esClient,
        values: secretsToCreate
      });
      let secretIndex = 0;
      if (tenantIdVar.value && !tenantIdVar.value.isSecretRef) {
        const tenantSecret = secrets[secretIndex];
        if (Array.isArray(tenantSecret)) {
          throw new _errors.CloudConnectorInvalidVarsError('Unexpected array of secrets for tenant_id');
        }
        tenantIdWithSecretRef = {
          ...tenantIdVar,
          value: {
            id: tenantSecret.id,
            isSecretRef: true
          }
        };
        secretIndex++;
      }
      if (clientIdVar.value && !clientIdVar.value.isSecretRef) {
        const clientSecret = secrets[secretIndex];
        if (Array.isArray(clientSecret)) {
          throw new _errors.CloudConnectorInvalidVarsError('Unexpected array of secrets for client_id');
        }
        clientIdWithSecretRef = {
          ...clientIdVar,
          value: {
            id: clientSecret.id,
            isSecretRef: true
          }
        };
      }
    }
    const azureCloudConnectorVars = {
      tenant_id: tenantIdWithSecretRef,
      client_id: clientIdWithSecretRef,
      azure_credentials_cloud_connector_id: azureCredentials
    };
    logger.debug(`Extracted Azure cloud connector vars: tenant_id=${!!tenantIdWithSecretRef}, client_id=${!!clientIdWithSecretRef}, azure_credentials=[REDACTED]`);
    return azureCloudConnectorVars;
  }
  logger.error(`Missing required Azure vars: tenant_id=${!!tenantIdVar}, client_id=${!!clientIdVar}, azure_credentials=[REDACTED]`);
  throw new _errors.CloudConnectorInvalidVarsError('Missing required Azure cloud connector variables: ' + `tenant_id=${!!tenantIdVar}, client_id=${!!clientIdVar}, azure_credentials=[REDACTED]`);
}