"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.incrementVersionAndUpdate = incrementVersionAndUpdate;
exports.updateCustomIntegration = updateCustomIntegration;
var _jsYaml = require("js-yaml");
var _constants = require("../../../../common/constants");
var _package_policy = require("../../package_policy");
var _archive_iterator = require("../archive/archive_iterator");
var _app_context = require("../../app_context");
var _errors = require("../../../errors");
var _get = require("./get");
var _install = require("./install");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Define a type for the integration attributes

async function updateCustomIntegration(esClient, soClient, id, fields) {
  // Get the current integration using the id
  let integration = null;
  try {
    integration = await soClient.get(_constants.PACKAGES_SAVED_OBJECT_TYPE, id);
  } catch (error) {
    // Ignore the error and handle the case where integration is null later
  }

  // if theres no integration, the soClient will throw the error above and we will ignore it so we can handle it with our own here
  if (!integration) {
    throw new _errors.CustomIntegrationNotFoundError(`Integration with ID ${id} not found`);
  } else if (integration.attributes.install_source !== 'custom' && integration.attributes.install_source !== 'upload') {
    throw new _errors.NotACustomIntegrationError(`Integration with ID ${id} is not a custom integration`);
  }

  // add one to the patch version in the semver
  const newVersion = integration.attributes.version.split('.');
  newVersion[2] = (parseInt(newVersion[2], 10) + 1).toString();
  const newVersionString = newVersion.join('.');
  // Increment the version of everything and create a new package
  const res = await incrementVersionAndUpdate(soClient, esClient, id, {
    version: newVersionString,
    readme: fields.readMeData,
    categories: fields.categories
  });
  return {
    version: newVersionString,
    status: res.status
  };
}
// Increments the version of everything, then creates a new package with the new version, readme, etc.
async function incrementVersionAndUpdate(soClient, esClient, pkgName, data) {
  const installedPkg = await (0, _get.getInstalledPackageWithAssets)({
    savedObjectsClient: soClient,
    pkgName
  });
  const assetsMap = [...installedPkg.assetsMap.entries()].reduce((acc, [path, content]) => {
    if (path === `${pkgName}-${installedPkg.installation.install_version}/manifest.yml`) {
      const yaml = (0, _jsYaml.load)(content.toString());
      yaml.version = data.version;
      content = Buffer.from((0, _jsYaml.dump)(yaml));
    }
    acc.set(path.replace(`-${installedPkg.installation.install_version}`, `-${data.version}`), content);
    return acc;
  }, new Map());

  // update the categories asset as well by editing the manifest.yml, but only if categories is actually defined in the request as its optional and we dont want to replace categories if they dont pass any in
  if (data.categories) {
    const manifestPath = `${pkgName}-${data.version}/manifest.yml`;
    const manifest = assetsMap.get(manifestPath);
    if (manifest) {
      const yaml = (0, _jsYaml.load)(manifest === null || manifest === void 0 ? void 0 : manifest.toString());
      if (yaml) {
        yaml.categories = data.categories || [];
        assetsMap.set(manifestPath, Buffer.from((0, _jsYaml.dump)(yaml)));
      }
    }
  }
  assetsMap.set(`${pkgName}-${data.version}/docs/README.md`, data.readme ? Buffer.from(data.readme) : Buffer.from(''));

  // update the changelog asset as well by adding an entry
  const changelogPath = `${pkgName}-${data.version}/changelog.yml`;
  const changelog = assetsMap.get(changelogPath);
  if (changelog) {
    const yaml = (0, _jsYaml.load)(changelog === null || changelog === void 0 ? void 0 : changelog.toString());
    if (yaml) {
      const newChangelogItem = {
        version: data.version,
        date: new Date().toISOString(),
        changes: [{
          type: 'update',
          description: `Edited integration and updated to version ${data.version}`,
          link: 'N/A'
        }]
      };
      yaml.push(newChangelogItem);
      assetsMap.set(changelogPath, Buffer.from((0, _jsYaml.dump)(yaml)));
    }
  }
  const paths = [...assetsMap.keys()];
  const packageInfo = {
    ...installedPkg.packageInfo,
    version: data.version
  };
  const archiveIterator = (0, _archive_iterator.createArchiveIteratorFromMap)(assetsMap);
  const packageInstallContext = {
    paths,
    packageInfo,
    archiveIterator
  };
  const res = await (0, _install.installPackageWithStateMachine)({
    packageInstallContext,
    pkgName,
    pkgVersion: data.version,
    installSource: 'custom',
    installType: 'install',
    savedObjectsClient: soClient,
    esClient,
    spaceId: 'default',
    force: true,
    paths: packageInstallContext.paths,
    authorizationHeader: null,
    keepFailedInstallation: true
  });
  const policyIdsToUpgrade = await _package_policy.packagePolicyService.listIds(_app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension(), {
    page: 1,
    perPage: _constants.SO_SEARCH_LIMIT,
    kuery: `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName}`
  });
  if (policyIdsToUpgrade.items.length) {
    await _package_policy.packagePolicyService.bulkUpgrade(soClient, esClient, policyIdsToUpgrade.items);
  }
  return res;
}