"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkRollbackAvailableCheck = bulkRollbackAvailableCheck;
exports.isIntegrationRollbackTTLExpired = void 0;
exports.rollbackAvailableCheck = rollbackAvailableCheck;
exports.rollbackInstallation = rollbackInstallation;
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _pMap = _interopRequireDefault(require("p-map"));
var _common = require("../../../../common");
var _errors = require("../../../errors");
var _bulk_handler = require("../../../routes/epm/bulk_handler");
var _ = require("../..");
var _upgrade_sender = require("../../upgrade_sender");
var _max_concurrency_constants = require("../../../constants/max_concurrency_constants");
var _get = require("./get");
var _install = require("./install");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_INTEGRATION_ROLLBACK_TTL = '7d';
const isIntegrationRollbackTTLExpired = installStartedAt => {
  var _appContextService$ge;
  let {
    integrationRollbackTTL
  } = (_appContextService$ge = _.appContextService.getConfig()) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : {};
  if (!integrationRollbackTTL) {
    integrationRollbackTTL = DEFAULT_INTEGRATION_ROLLBACK_TTL;
  }
  const numberPart = integrationRollbackTTL.slice(0, -1);
  const unitPart = integrationRollbackTTL.slice(-1);
  const ttlDuration = _moment.default.duration(Number(numberPart), unitPart).asMilliseconds();
  return Date.parse(installStartedAt) < Date.now() - ttlDuration;
};
exports.isIntegrationRollbackTTLExpired = isIntegrationRollbackTTLExpired;
async function rollbackAvailableCheck(pkgName, currentUserPolicyIds) {
  // Need a less restrictive client than fleetContext.internalSoClient for SO operations in multiple spaces.
  const savedObjectsClient = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const packageSORes = await (0, _get.getPackageSavedObjects)(savedObjectsClient, {
    searchFields: ['name'],
    search: pkgName,
    fields: ['version', 'previous_version', 'install_started_at', 'install_source']
  });
  if (packageSORes.saved_objects.length === 0) {
    return {
      isAvailable: false,
      reason: `Package ${pkgName} not found`
    };
  } else if (packageSORes.saved_objects.length > 1) {
    // This should not happen.
    return {
      isAvailable: false,
      reason: `Expected exactly one package saved object`
    };
  }
  const packageSO = packageSORes.saved_objects[0];
  const previousVersion = packageSO.attributes.previous_version;
  if (!previousVersion) {
    return {
      isAvailable: false,
      reason: `No previous version found for package ${pkgName}`
    };
  }
  if (isIntegrationRollbackTTLExpired(packageSO.attributes.install_started_at)) {
    return {
      isAvailable: false,
      reason: `Rollback not allowed as TTL expired`
    };
  }
  if (packageSO.attributes.install_source !== 'registry') {
    return {
      isAvailable: false,
      reason: `${pkgName} was not installed from the registry (install source: ${packageSO.attributes.install_source})`
    };
  }
  const packagePolicySORes = await _.packagePolicyService.getPackagePolicySavedObjects(savedObjectsClient, {
    searchFields: ['package.name'],
    search: pkgName,
    spaceIds: ['*'],
    fields: ['package.version', 'is_managed', 'policy_ids']
  });
  const packagePolicySOs = packagePolicySORes.saved_objects;
  const policyIds = packagePolicySOs.map(so => so.id);
  const managedRollbackReason = `Cannot rollback integration with managed package policies`;
  if (packagePolicySOs.some(so => so.attributes.is_managed)) {
    return {
      isAvailable: false,
      reason: managedRollbackReason
    };
  }
  // checking is_managed flag on agent policy, it is not always set on package policy
  const agentPolicyIds = (0, _lodash.uniq)(packagePolicySOs.flatMap(so => {
    var _so$attributes$policy;
    return (_so$attributes$policy = so.attributes.policy_ids) !== null && _so$attributes$policy !== void 0 ? _so$attributes$policy : [];
  }));
  const agentPolicies = await _.agentPolicyService.getByIds(savedObjectsClient, agentPolicyIds.map(id => ({
    id,
    spaceId: '*'
  })));
  if (agentPolicies.some(agentPolicy => agentPolicy.is_managed)) {
    return {
      isAvailable: false,
      reason: managedRollbackReason
    };
  }
  const packageVersion = packageSO === null || packageSO === void 0 ? void 0 : packageSO.attributes.version;
  if (packagePolicySOs.filter(so => !so.id.endsWith(':prev')).some(so => {
    var _so$attributes$packag;
    return ((_so$attributes$packag = so.attributes.package) === null || _so$attributes$packag === void 0 ? void 0 : _so$attributes$packag.version) !== packageVersion;
  })) {
    return {
      isAvailable: false,
      reason: `Rollback not available because some integration policies are not upgraded to version ${packageVersion}`
    };
  }
  if (packagePolicySOs.length > 0) {
    const policyIdsWithNoPreviousVersion = policyIds.filter(soId => {
      if (!soId.endsWith(':prev')) {
        return !policyIds.includes(`${soId}:prev`);
      }
      return false;
    });
    if (policyIdsWithNoPreviousVersion.length > 0) {
      return {
        isAvailable: false,
        reason: `No previous version found for package policies: ${policyIdsWithNoPreviousVersion.join(', ')}`
      };
    }
    const policiesOnWrongPreviousVersion = packagePolicySOs.filter(so => {
      if (so.id.endsWith(':prev')) {
        var _so$attributes$packag2;
        return ((_so$attributes$packag2 = so.attributes.package) === null || _so$attributes$packag2 === void 0 ? void 0 : _so$attributes$packag2.version) !== previousVersion;
      }
      return false;
    });
    if (policiesOnWrongPreviousVersion.length > 0) {
      return {
        isAvailable: false,
        reason: `Rollback not available because not all integration policies were upgraded from the same previous version ${previousVersion}`
      };
    }
  }
  if (currentUserPolicyIds.length < policyIds.length) {
    return {
      isAvailable: false,
      reason: `Not authorized to rollback integration policies in all spaces`
    };
  }
  return {
    isAvailable: true
  };
}
async function bulkRollbackAvailableCheck(request) {
  const savedObjectsClient = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const result = await savedObjectsClient.find({
    type: _common.PACKAGES_SAVED_OBJECT_TYPE,
    fields: ['name'],
    filter: `${_common.PACKAGES_SAVED_OBJECT_TYPE}.attributes.install_status:installed`,
    perPage: _common.SO_SEARCH_LIMIT
  });
  const installedPackageNames = result.saved_objects.map(so => so.attributes.name);
  const items = {};
  const packagePolicyIdsForCurrentUser = await (0, _bulk_handler.getPackagePolicyIdsForCurrentUser)(request, installedPackageNames.map(name => ({
    name
  })));
  await (0, _pMap.default)(installedPackageNames, async pkgName => {
    const {
      isAvailable,
      reason
    } = await rollbackAvailableCheck(pkgName, packagePolicyIdsForCurrentUser[pkgName]);
    items[pkgName] = {
      isAvailable,
      reason
    };
  }, {
    concurrency: _max_concurrency_constants.MAX_CONCURRENT_EPM_PACKAGES_INSTALLATIONS
  });
  return items;
}
async function rollbackInstallation(options) {
  const {
    esClient,
    currentUserPolicyIds,
    pkgName,
    spaceId
  } = options;
  const logger = _.appContextService.getLogger();
  logger.info(`Starting installation rollback for package: ${pkgName}`);

  // Need a less restrictive client than fleetContext.internalSoClient for SO operations in multiple spaces.
  const savedObjectsClient = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  const {
    isAvailable,
    reason
  } = await rollbackAvailableCheck(pkgName, currentUserPolicyIds);
  if (!isAvailable) {
    throw new _errors.PackageRollbackError(reason ? reason : `Rollback not available for package ${pkgName}`);
  }

  // Retrieve the package saved object, throw if it doesn't exist or doesn't have a previous version.
  const packageSORes = await (0, _get.getPackageSavedObjects)(savedObjectsClient, {
    searchFields: ['name'],
    search: pkgName
  });
  const packageSO = packageSORes.saved_objects[0];
  const previousVersion = packageSO.attributes.previous_version;
  logger.info(`Rolling back ${pkgName} from ${packageSO.attributes.version} to ${previousVersion}`);

  // Retrieve package policy saved objects, throw if any of them doesn't have a previous version or has a different one.
  const packagePolicySORes = await _.packagePolicyService.getPackagePolicySavedObjects(savedObjectsClient, {
    searchFields: ['package.name'],
    search: pkgName,
    spaceIds: ['*']
  });
  const packagePolicySOs = packagePolicySORes.saved_objects;
  try {
    // Roll back package policies.
    const rollbackResult = await _.packagePolicyService.rollback(savedObjectsClient, packagePolicySOs);

    // Roll back package.
    const res = await (0, _install.installPackage)({
      esClient,
      savedObjectsClient,
      installSource: 'registry',
      // Can only rollback from the registry.
      pkgkey: `${pkgName}-${previousVersion}`,
      spaceId,
      force: true
    });
    if (res.error) {
      await _.packagePolicyService.restoreRollback(savedObjectsClient, rollbackResult);
      throw new _errors.PackageRollbackError(`Failed to rollback package ${pkgName} to version ${previousVersion}: ${res.error.message}`);
    }

    // Clean up package policies previous revisions and package policies that were copied during rollback.
    await _.packagePolicyService.cleanupRollbackSavedObjects(savedObjectsClient, rollbackResult);
    // Bump agent policy revision for all package policies that were rolled back.
    await _.packagePolicyService.bumpAgentPolicyRevisionAfterRollback(savedObjectsClient, rollbackResult);
  } catch (error) {
    sendRollbackTelemetry({
      packageName: pkgName,
      currentVersion: packageSO.attributes.version,
      previousVersion,
      success: false,
      errorMessage: error.message
    });
    throw error;
  }
  sendRollbackTelemetry({
    packageName: pkgName,
    currentVersion: packageSO.attributes.version,
    previousVersion,
    success: true
  });
  logger.info(`Package: ${pkgName} successfully rolled back to version: ${previousVersion}`);
  return {
    version: previousVersion,
    success: true
  };
}
function sendRollbackTelemetry({
  packageName,
  currentVersion,
  previousVersion,
  success,
  errorMessage
}) {
  const upgradeTelemetry = {
    packageName,
    currentVersion,
    newVersion: previousVersion,
    status: success ? 'success' : 'failure',
    eventType: _upgrade_sender.UpdateEventType.PACKAGE_ROLLBACK,
    errorMessage
  };
  (0, _upgrade_sender.sendTelemetryEvents)(_.appContextService.getLogger(), _.appContextService.getTelemetryEventsSender(), upgradeTelemetry);
  _.appContextService.getLogger().debug(`Send rollback telemetry: ${JSON.stringify(upgradeTelemetry)}`);
}