"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cloudConnectorService = exports.CloudConnectorService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("../../common/constants");
var _cloud_connector = require("../../common/constants/cloud_connector");
var _errors = require("../errors");
var _app_context = require("./app_context");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
class CloudConnectorService {
  getLogger(...childContextPaths) {
    return _app_context.appContextService.getLogger().get('CloudConnectorService', ...childContextPaths);
  }

  /**
   * Normalizes a cloud connector name by trimming and collapsing consecutive spaces
   * @param name - The name to normalize
   * @returns The normalized name
   */
  static normalizeName(name) {
    return name.trim().replace(/\s+/g, ' ');
  }

  /**
   * Validates and normalizes a cloud connector name, checking for duplicates
   * @param soClient - Saved objects client
   * @param name - The name to validate
   * @param excludeId - Optional cloud connector ID to exclude from duplicate check (for updates)
   * @returns The normalized name
   * @throws CloudConnectorCreateError if a duplicate name is found
   */
  async validateAndNormalizeName(soClient, name, excludeId) {
    const normalizedName = CloudConnectorService.normalizeName(name);

    // Check for existing connector with same name (case-insensitive, normalized)
    const existingConnectors = await this.getList(soClient, {
      perPage: _constants.SO_SEARCH_LIMIT,
      fields: ['name']
    });
    const normalizedNameLower = normalizedName.toLowerCase();
    const duplicateConnectorName = existingConnectors.find(c => {
      // Skip the current connector when updating
      if (excludeId && c.id === excludeId) {
        return false;
      }
      return CloudConnectorService.normalizeName(c.name).toLowerCase() === normalizedNameLower;
    });
    if (duplicateConnectorName) {
      throw new _errors.CloudConnectorCreateError('A cloud connector with this name already exists');
    }
    return normalizedName;
  }
  async create(soClient, cloudConnector) {
    const logger = this.getLogger('create');
    try {
      logger.info('Creating cloud connector');
      this.validateCloudConnectorDetails(cloudConnector);
      const {
        vars,
        cloudProvider
      } = cloudConnector;
      if (!vars || Object.keys(vars).length === 0) {
        logger.error(`Package policy must contain ${cloudProvider} input vars`);
        throw new _errors.CloudConnectorCreateError(`CloudConnectorService Package policy must contain ${cloudProvider} input vars`);
      }

      // Validate and normalize the name, checking for duplicates
      const name = await this.validateAndNormalizeName(soClient, cloudConnector.name);

      // Check if space awareness is enabled for namespace handling
      const {
        isSpaceAwarenessEnabled
      } = await Promise.resolve().then(() => _interopRequireWildcard(require('./spaces/helpers')));
      const useSpaceAwareness = await isSpaceAwarenessEnabled();
      const namespace = useSpaceAwareness ? '*' : undefined;
      const cloudConnectorAttributes = {
        name,
        namespace,
        cloudProvider,
        accountType: cloudConnector.accountType,
        vars,
        packagePolicyCount: 1,
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString()
      };
      const savedObject = await soClient.create(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorAttributes);
      logger.info('Successfully created cloud connector');
      return {
        id: savedObject.id,
        ...savedObject.attributes
      };
    } catch (error) {
      logger.error(`Failed to create cloud connector: ${error instanceof Error ? error.message : String(error)}`);
      (0, _errors.rethrowIfInstanceOrWrap)(error, _errors.CloudConnectorCreateError, 'CloudConnectorService Failed to create cloud connector');
    }
  }
  async getList(soClient, options) {
    const logger = this.getLogger('getList');
    logger.debug('Getting cloud connectors list');
    try {
      const findOptions = {
        type: _constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE,
        page: (options === null || options === void 0 ? void 0 : options.page) || 1,
        perPage: (options === null || options === void 0 ? void 0 : options.perPage) || 20,
        sortField: 'created_at',
        sortOrder: 'desc'
      };

      // Add KQL filter if specified
      if (options !== null && options !== void 0 && options.kuery) {
        findOptions.filter = options.kuery;
      }

      // Add fields filter if specified
      if (options !== null && options !== void 0 && options.fields) {
        findOptions.fields = options.fields;
      }
      const cloudConnectors = await soClient.find(findOptions);
      logger.debug('Successfully retrieved cloud connectors list');
      return cloudConnectors.saved_objects.map(savedObject => ({
        id: savedObject.id,
        ...savedObject.attributes
      }));
    } catch (error) {
      logger.error('Failed to get cloud connectors list', error.message);
      throw new _errors.CloudConnectorGetListError(`Failed to get cloud connectors list: ${error.message}\n${error.stack}`);
    }
  }
  async getById(soClient, cloudConnectorId) {
    const logger = this.getLogger('getById');
    try {
      logger.info(`Getting cloud connector ${cloudConnectorId}`);
      const cloudConnector = await soClient.get(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);
      logger.info(`Successfully retrieved cloud connector ${cloudConnectorId}`);
      return {
        id: cloudConnector.id,
        ...cloudConnector.attributes
      };
    } catch (error) {
      logger.error('Failed to get cloud connector', error.message);
      throw new _errors.CloudConnectorGetListError(`Failed to get cloud connector: ${error.message}\n${error.stack}`);
    }
  }
  async update(soClient, cloudConnectorId, cloudConnectorUpdate) {
    const logger = this.getLogger('update');
    try {
      logger.info(`Updating cloud connector ${cloudConnectorId}`);

      // Get existing cloud connector
      const existingCloudConnector = await soClient.get(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);

      // Validate updates if vars are provided
      if (cloudConnectorUpdate.vars) {
        const tempCloudConnector = {
          name: cloudConnectorUpdate.name || existingCloudConnector.attributes.name,
          vars: cloudConnectorUpdate.vars,
          cloudProvider: existingCloudConnector.attributes.cloudProvider
        };
        this.validateCloudConnectorDetails(tempCloudConnector);
      }

      // Prepare update attributes
      const updateAttributes = {
        updated_at: new Date().toISOString()
      };

      // Validate and normalize name if provided, checking for duplicates (excluding current connector)
      if (cloudConnectorUpdate.name) {
        updateAttributes.name = await this.validateAndNormalizeName(soClient, cloudConnectorUpdate.name, cloudConnectorId);
      }
      if (cloudConnectorUpdate.accountType !== undefined) {
        updateAttributes.accountType = cloudConnectorUpdate.accountType;
      }
      if (cloudConnectorUpdate.vars) {
        updateAttributes.vars = cloudConnectorUpdate.vars;
      }

      // Update the saved object
      const updatedSavedObject = await soClient.update(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId, updateAttributes);
      logger.info(`Successfully updated cloud connector ${cloudConnectorId}`);

      // Return the updated cloud connector with merged attributes
      const mergedAttributes = {
        ...existingCloudConnector.attributes,
        ...updatedSavedObject.attributes
      };
      return {
        id: cloudConnectorId,
        ...mergedAttributes
      };
    } catch (error) {
      logger.error(`Failed to update cloud connector: ${error instanceof Error ? error.message : String(error)}`);
      (0, _errors.rethrowIfInstanceOrWrap)(error, _errors.CloudConnectorCreateError, 'Failed to update cloud connector');
    }
  }
  async delete(soClient, cloudConnectorId, force = false) {
    const logger = this.getLogger('delete');
    try {
      logger.info(`Deleting cloud connector ${cloudConnectorId} (force: ${force})`);

      // First, get the cloud connector to check packagePolicyCount
      const cloudConnector = await soClient.get(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);

      // Check if cloud connector is still in use by package policies (unless force is true)
      if (!force && cloudConnector.attributes.packagePolicyCount > 0) {
        const errorMessage = `Cannot delete cloud connector "${cloudConnector.attributes.name}" as it is being used by ${cloudConnector.attributes.packagePolicyCount} package policies`;
        logger.error(errorMessage);
        throw new _errors.CloudConnectorDeleteError(errorMessage);
      }

      // Log a warning if force deleting a connector that's still in use
      if (force && cloudConnector.attributes.packagePolicyCount > 0) {
        logger.warn(`Force deleting cloud connector "${cloudConnector.attributes.name}" which is still being used by ${cloudConnector.attributes.packagePolicyCount} package policies`);
      }

      // Delete the cloud connector
      await soClient.delete(_constants.CLOUD_CONNECTOR_SAVED_OBJECT_TYPE, cloudConnectorId);
      logger.info(`Successfully deleted cloud connector ${cloudConnectorId}`);
      return {
        id: cloudConnectorId
      };
    } catch (error) {
      logger.error('Failed to delete cloud connector', error.message);

      // Re-throw CloudConnectorDeleteError as-is to preserve the original error message
      if (error instanceof _errors.CloudConnectorDeleteError) {
        throw error;
      }
      throw new _errors.CloudConnectorDeleteError(`Failed to delete cloud connector: ${error.message}\n${error.stack}`);
    }
  }
  validateCloudConnectorDetails(cloudConnector) {
    const logger = this.getLogger('validate cloud connector details');
    const vars = cloudConnector.vars;
    if (cloudConnector.cloudProvider === 'aws') {
      var _awsVars$role_arn, _awsVars$external_id;
      // Type assertion is safe here because we perform runtime validation below
      const awsVars = vars;
      const roleArn = (_awsVars$role_arn = awsVars.role_arn) === null || _awsVars$role_arn === void 0 ? void 0 : _awsVars$role_arn.value;
      if (!roleArn) {
        logger.error('Package policy must contain role_arn variable');
        throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain role_arn variable');
      }
      const externalId = (_awsVars$external_id = awsVars.external_id) === null || _awsVars$external_id === void 0 ? void 0 : _awsVars$external_id.value;
      if (!externalId) {
        logger.error('Package policy must contain valid external_id secret reference');
        throw new _errors.CloudConnectorInvalidVarsError('Package policy must contain valid external_id secret reference');
      }
      const isValidExternalId = (externalId === null || externalId === void 0 ? void 0 : externalId.id) && (externalId === null || externalId === void 0 ? void 0 : externalId.isSecretRef) && CloudConnectorService.EXTERNAL_ID_REGEX.test(externalId.id);
      if (!isValidExternalId) {
        logger.error('External ID secret reference must be a valid secret reference');
        throw new _errors.CloudConnectorInvalidVarsError('External ID secret reference is not valid');
      }
    } else if (cloudConnector.cloudProvider === 'azure') {
      var _tenantId$value, _tenantId$value2, _clientId$value, _clientId$value2;
      // Type assertion is safe here because we perform runtime validation below
      const azureVars = vars;
      // Validate that all required Azure fields have valid secret references
      const tenantId = azureVars.tenant_id;
      const clientId = azureVars.client_id;
      const azureCredentials = azureVars.azure_credentials_cloud_connector_id;
      if (!(tenantId !== null && tenantId !== void 0 && (_tenantId$value = tenantId.value) !== null && _tenantId$value !== void 0 && _tenantId$value.id) || !(tenantId !== null && tenantId !== void 0 && (_tenantId$value2 = tenantId.value) !== null && _tenantId$value2 !== void 0 && _tenantId$value2.isSecretRef)) {
        logger.error(`Package policy must contain valid ${_cloud_connector.TENANT_ID_VAR_NAME} secret reference`);
        throw new _errors.CloudConnectorInvalidVarsError(`${_cloud_connector.TENANT_ID_VAR_NAME} must be a valid secret reference`);
      }
      if (!(clientId !== null && clientId !== void 0 && (_clientId$value = clientId.value) !== null && _clientId$value !== void 0 && _clientId$value.id) || !(clientId !== null && clientId !== void 0 && (_clientId$value2 = clientId.value) !== null && _clientId$value2 !== void 0 && _clientId$value2.isSecretRef)) {
        logger.error(`Package policy must contain valid ${_cloud_connector.CLIENT_ID_VAR_NAME} secret reference`);
        throw new _errors.CloudConnectorInvalidVarsError(`${_cloud_connector.CLIENT_ID_VAR_NAME} must be a valid secret reference`);
      }
      if (!(azureCredentials !== null && azureCredentials !== void 0 && azureCredentials.value)) {
        logger.error(`Package policy must contain valid ${_cloud_connector.AZURE_CREDENTIALS_CLOUD_CONNECTOR_ID} value`);
        throw new _errors.CloudConnectorInvalidVarsError(`${_cloud_connector.AZURE_CREDENTIALS_CLOUD_CONNECTOR_ID} must be a valid string`);
      }
    } else {
      logger.error(`Unsupported cloud provider: ${cloudConnector.cloudProvider}`);
      throw new _errors.CloudConnectorCreateError(`Unsupported cloud provider: ${cloudConnector.cloudProvider}`);
    }
  }
}
exports.CloudConnectorService = CloudConnectorService;
(0, _defineProperty2.default)(CloudConnectorService, "EXTERNAL_ID_REGEX", /^[a-zA-Z0-9_-]{20}$/);
const cloudConnectorService = exports.cloudConnectorService = new CloudConnectorService();