"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RollbackActionRunner = void 0;
exports.rollbackBatch = rollbackBatch;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _app_context = require("../app_context");
var _errors = require("../../errors");
var _action_runner = require("./action_runner");
var _bulk_action_types = require("./bulk_action_types");
var _hosted_agent = require("./hosted_agent");
var _actions = require("./actions");
var _crud = require("./crud");
var _rollback = require("./rollback");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RollbackActionRunner extends _action_runner.ActionRunner {
  constructor(...args) {
    super(...args);
    (0, _defineProperty2.default)(this, "allActionIds", []);
  }
  async processAgents(agents) {
    var _this$actionParams, _this$actionParams2;
    const result = await rollbackBatch(this.esClient, agents, {}, {
      // Don't pass actionId - each version group should get its own unique ID
      // actionId is for tracking the bulk operation task, not individual rollback actions
      // Don't pass total - each action should use the count for its version group, not the total across all batches
      spaceIds: (_this$actionParams = this.actionParams) !== null && _this$actionParams !== void 0 && _this$actionParams.spaceId ? [this.actionParams.spaceId] : undefined
    }, (_this$actionParams2 = this.actionParams) !== null && _this$actionParams2 !== void 0 && _this$actionParams2.spaceId ? [this.actionParams.spaceId] : undefined);
    // Track all action IDs created across batches
    this.allActionIds.push(...result.actionIds);
    // Return the first actionId for compatibility with ActionRunner interface
    // If no actions were created, return empty string
    return {
      actionId: result.actionIds.length > 0 ? result.actionIds[0] : ''
    };
  }
  async processAgentsInBatches() {
    // Reset action IDs tracking for this batch processing run
    this.allActionIds = [];
    const result = await super.processAgentsInBatches();
    // Store all collected action IDs in actionParams for retrieval
    if (this.allActionIds.length > 0) {
      this.actionParams.allActionIds = this.allActionIds;
    }
    return result;
  }
  getAllActionIds() {
    return this.allActionIds;
  }
  getTaskType() {
    return _bulk_action_types.BulkActionTaskType.ROLLBACK_RETRY;
  }
  getActionType() {
    return 'UPGRADE';
  }
}
exports.RollbackActionRunner = RollbackActionRunner;
async function rollbackBatch(esClient, givenAgents, outgoingErrors, options, spaceIds) {
  var _ref;
  const soClient = _app_context.appContextService.getInternalUserSOClientForSpaceId(spaceIds === null || spaceIds === void 0 ? void 0 : spaceIds[0]);
  const hostedPolicies = await (0, _hosted_agent.getHostedPolicies)(soClient, givenAgents);

  // Eligible agents, grouped by rollback version
  const agentsByRollbackVersion = {};

  // Pre-action errors are not linked to a specific rollback version:
  // - agent not found
  // - agent exists but has no available rollback
  const preActionErrors = {
    ...outgoingErrors
  };
  // Version-specific errors are linked to a specific rollback version:
  // - expired rollback
  // - hosted agent
  const errorsByRollbackVersion = {};
  for (const agent of givenAgents) {
    var _agent$upgrade;
    const rollbacks = ((_agent$upgrade = agent.upgrade) === null || _agent$upgrade === void 0 ? void 0 : _agent$upgrade.rollbacks) || [];

    // If agent has no available rollbacks, add to pre-action errors
    if (rollbacks.length === 0) {
      preActionErrors[agent.id] = new _errors.AgentRollbackError(_rollback.NO_ROLLBACK_ERROR_MESSAGE);
      continue;
    }
    const rollbackVersion = rollbacks[0].version;

    // If agent has expired rollback, add to version-specific errors
    const validRollbacks = (0, _rollback.getValidRollbacks)(agent);
    if (validRollbacks.length === 0) {
      if (!errorsByRollbackVersion[rollbackVersion]) {
        errorsByRollbackVersion[rollbackVersion] = {};
      }
      errorsByRollbackVersion[rollbackVersion][agent.id] = new _errors.AgentRollbackError(_rollback.EXPIRED_ROLLBACK_ERROR_MESSAGE);
      continue;
    }

    // If agent is hosted, add to version-specific errors
    if ((0, _hosted_agent.isHostedAgent)(hostedPolicies, agent)) {
      if (!errorsByRollbackVersion[rollbackVersion]) {
        errorsByRollbackVersion[rollbackVersion] = {};
      }
      errorsByRollbackVersion[rollbackVersion][agent.id] = new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot rollback agent in hosted agent policy ${agent.policy_id}`);
      continue;
    }

    // Agent is eligible for upgrade rollback
    if (!agentsByRollbackVersion[rollbackVersion]) {
      agentsByRollbackVersion[rollbackVersion] = [];
    }
    agentsByRollbackVersion[rollbackVersion].push(agent);
  }

  // Create upgrade actions with rollback:true for each version group with agents
  const actionIds = [];
  const errorsByActionId = {};
  const now = new Date().toISOString();
  const namespaces = (_ref = spaceIds !== null && spaceIds !== void 0 ? spaceIds : options.spaceIds) !== null && _ref !== void 0 ? _ref : [];
  for (const [version, agents] of Object.entries(agentsByRollbackVersion)) {
    var _options$actionId, _options$total;
    const actionId = (_options$actionId = options.actionId) !== null && _options$actionId !== void 0 ? _options$actionId : (0, _uuid.v4)();
    const total = (_options$total = options.total) !== null && _options$total !== void 0 ? _options$total : agents.length;
    const data = {
      version,
      rollback: true
    };
    await (0, _actions.createAgentAction)(esClient, soClient, {
      id: actionId,
      created_at: now,
      data,
      ack_data: data,
      type: 'UPGRADE',
      total,
      agents: agents.map(agent => agent.id),
      namespaces
    });
    actionIds.push(actionId);
    errorsByActionId[actionId] = {};

    // Assign version-specific errors (expired rollbacks, hosted agents) to this action
    if (errorsByRollbackVersion[version]) {
      Object.assign(errorsByActionId[actionId], errorsByRollbackVersion[version]);
    }
    await (0, _crud.bulkUpdateAgents)(esClient, agents.map(agent => ({
      agentId: agent.id,
      data: {
        upgraded_at: null,
        upgrade_started_at: now
      }
    })), errorsByActionId[actionId]);
  }

  // Create actions for versions that have errors but no eligible agents
  for (const [version, versionErrors] of Object.entries(errorsByRollbackVersion)) {
    // Skip if we already created an action for this version
    if (agentsByRollbackVersion[version]) {
      continue;
    }

    // Create error-only action for this version
    const actionId = (0, _uuid.v4)();
    const total = Object.keys(versionErrors).length;
    await (0, _actions.createAgentAction)(esClient, soClient, {
      id: actionId,
      created_at: now,
      data: {
        version,
        rollback: true
      },
      ack_data: {
        version,
        rollback: true
      },
      type: 'UPGRADE',
      total,
      agents: [],
      namespaces
    });
    actionIds.push(actionId);
    errorsByActionId[actionId] = {
      ...versionErrors
    };
  }

  // Create error action results

  // Pre-action errors cannot be associated with a specific rollback version. These errors
  // are assigned to the first action, or to an error-only action if no valid actions were created.
  if (Object.keys(preActionErrors).length > 0) {
    if (actionIds.length > 0) {
      if (!errorsByActionId[actionIds[0]]) {
        errorsByActionId[actionIds[0]] = {};
      }
      Object.assign(errorsByActionId[actionIds[0]], preActionErrors);
    } else {
      const errorOnlyActionId = (0, _uuid.v4)();
      await (0, _actions.createAgentAction)(esClient, soClient, {
        id: errorOnlyActionId,
        created_at: now,
        data: {
          rollback: true
        },
        ack_data: {
          rollback: true
        },
        type: 'UPGRADE',
        total: Object.keys(preActionErrors).length,
        agents: [],
        namespaces
      });
      actionIds.push(errorOnlyActionId);
      errorsByActionId[errorOnlyActionId] = preActionErrors;
    }
  }

  // Errors for agents with rollback data are assigned to their version-specific action.
  for (const actionId of actionIds) {
    const actionErrors = errorsByActionId[actionId];
    if (actionErrors && Object.keys(actionErrors).length > 0) {
      await (0, _actions.createErrorActionResults)(esClient, actionId, actionErrors, 'no valid rollback available for agent');
    }
  }
  return {
    actionIds
  };
}