"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _constants = require("../../../common/constants");
var _api_privileges = require("../../constants/api_privileges");
var _errors = require("../schema/errors");
var _cloud_connector = require("../../types/rest_spec/cloud_connector");
var _handlers = require("./handlers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = router => {
  // POST /api/fleet/cloud_connectors
  router.versioned.post({
    path: _constants.CLOUD_CONNECTOR_API_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
        }]
      }
    },
    summary: 'Create cloud connector',
    options: {
      tags: ['oas-tag:Fleet cloud connectors'],
      availability: {
        since: '9.2.0',
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _cloud_connector.CreateCloudConnectorRequestSchema,
      response: {
        200: {
          body: () => _cloud_connector.CreateCloudConnectorResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.createCloudConnectorHandler);

  // GET /api/fleet/cloud_connectors
  router.versioned.get({
    path: _constants.CLOUD_CONNECTOR_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.READ]
        }]
      }
    },
    summary: 'Get cloud connectors',
    options: {
      tags: ['oas-tag:Fleet cloud connectors'],
      availability: {
        since: '9.2.0',
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _cloud_connector.GetCloudConnectorsRequestSchema,
      response: {
        200: {
          body: () => _cloud_connector.GetCloudConnectorsResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.getCloudConnectorsHandler);

  // GET /api/fleet/cloud_connectors/{cloudConnectorId}
  router.versioned.get({
    path: _constants.CLOUD_CONNECTOR_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.READ]
        }]
      }
    },
    summary: 'Get cloud connector',
    options: {
      tags: ['oas-tag:Fleet cloud connectors'],
      availability: {
        since: '9.2.0',
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _cloud_connector.GetCloudConnectorRequestSchema,
      response: {
        200: {
          body: () => _cloud_connector.GetCloudConnectorResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.getCloudConnectorHandler);

  // PUT /api/fleet/cloud_connectors/{cloudConnectorId}
  router.versioned.put({
    path: _constants.CLOUD_CONNECTOR_API_ROUTES.UPDATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
        }]
      }
    },
    summary: 'Update cloud connector',
    options: {
      tags: ['oas-tag:Fleet cloud connectors'],
      availability: {
        since: '9.2.0',
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _cloud_connector.UpdateCloudConnectorRequestSchema,
      response: {
        200: {
          body: () => _cloud_connector.UpdateCloudConnectorResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.updateCloudConnectorHandler);

  // DELETE /api/fleet/cloud_connectors/{cloudConnectorId}
  router.versioned.delete({
    path: _constants.CLOUD_CONNECTOR_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
        }]
      }
    },
    summary: 'Delete cloud connector (supports force deletion)',
    options: {
      tags: ['oas-tag:Fleet cloud connectors'],
      availability: {
        since: '9.2.0',
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _cloud_connector.DeleteCloudConnectorRequestSchema,
      response: {
        200: {
          body: () => _cloud_connector.DeleteCloudConnectorResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.deleteCloudConnectorHandler);

  // GET /api/fleet/cloud_connectors/{cloudConnectorId}/usage
  router.versioned.get({
    path: _constants.CLOUD_CONNECTOR_API_ROUTES.USAGE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.READ]
        }]
      }
    },
    summary: 'Get cloud connector usage (package policies using the connector)',
    options: {
      tags: ['oas-tag:Fleet cloud connectors'],
      availability: {
        since: '9.2.0',
        stability: 'experimental'
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    options: {
      oasOperationObject: () => ({
        responses: {
          '200': {
            content: {
              'application/json': {
                examples: {
                  getCloudConnectorUsageResponseExample: {
                    description: 'Example response showing package policies using the cloud connector',
                    value: {
                      items: [{
                        id: 'package-policy-1',
                        name: 'CSPM AWS Policy',
                        package: {
                          name: 'cloud_security_posture',
                          title: 'Cloud Security Posture Management',
                          version: '3.1.1'
                        },
                        policy_ids: ['policy-id-123', 'policy-id-456'],
                        created_at: '2025-01-16T09:00:00.000Z',
                        updated_at: '2025-01-16T09:00:00.000Z'
                      }],
                      total: 2,
                      page: 1,
                      perPage: 20
                    }
                  }
                }
              }
            }
          },
          '400': {
            content: {
              'application/json': {
                examples: {
                  genericErrorResponseExample: {
                    description: 'Example of a generic error response',
                    value: {
                      statusCode: 400,
                      error: 'Bad Request',
                      message: 'Cloud connector not found'
                    }
                  }
                }
              }
            }
          }
        }
      })
    },
    validate: {
      request: _cloud_connector.GetCloudConnectorUsageRequestSchema,
      response: {
        200: {
          body: () => _cloud_connector.GetCloudConnectorUsageResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.getCloudConnectorUsageHandler);
};
exports.registerRoutes = registerRoutes;