"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndicesList = exports.getESQLSources = void 0;
var _esqlTypes = require("@kbn/esql-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const INTEGRATIONS_API = '/api/fleet/epm/packages/installed';
const API_VERSION = '2023-10-31';
/**
 * Fetches the list of indices, aliases, and data streams from the Elasticsearch cluster.
 * @param core The core start contract to make HTTP requests.
 * @param areRemoteIndicesAvailable A boolean indicating if remote indices should be included.
 * @returns A promise that resolves to an array of ESQLSourceResult objects.
 */
const getIndicesList = async (core, areRemoteIndicesAvailable) => {
  const scope = areRemoteIndicesAvailable ? 'all' : 'local';
  const response = await core.http.get(`${_esqlTypes.SOURCES_AUTOCOMPLETE_ROUTE}${scope}`).catch(error => {
    // eslint-disable-next-line no-console
    console.error('Failed to fetch the sources', error);
    return [];
  });
  return response;
};

/**
 * Fetches integrations with active data streams from the Fleet API
 * @param core The core start contract to make HTTP requests and access application capabilities.
 * @returns A promise that resolves to an array of ESQLSourceResult objects.
 */
exports.getIndicesList = getIndicesList;
const getIntegrations = async core => {
  var _response$items$filte, _response$items;
  const fleetCapabilities = core.application.capabilities.fleet;
  if (!(fleetCapabilities !== null && fleetCapabilities !== void 0 && fleetCapabilities.read)) {
    return [];
  }
  // Ideally we should use the Fleet plugin constants to fetch the integrations
  // import { EPM_API_ROUTES, API_VERSIONS } from '@kbn/fleet-plugin/common';
  // but it complicates things as we need to use an x-pack plugin as dependency to get 2 constants
  // and this needs to be done in various places in the codebase which use the editor
  // https://github.com/elastic/kibana/issues/186061
  const response = await core.http.get(INTEGRATIONS_API, {
    query: {
      showOnlyActiveDataStreams: true
    },
    version: API_VERSION
  }).catch(error => {
    // eslint-disable-next-line no-console
    console.error('Failed to fetch integrations', error);
  });
  return (_response$items$filte = response === null || response === void 0 ? void 0 : (_response$items = response.items) === null || _response$items === void 0 ? void 0 : _response$items.filter(({
    dataStreams
  }) => dataStreams.length).map(source => ({
    name: source.name,
    hidden: false,
    title: source.title,
    dataStreams: source.dataStreams,
    type: _esqlTypes.SOURCES_TYPES.INTEGRATION
  }))) !== null && _response$items$filte !== void 0 ? _response$items$filte : [];
};

/** Fetches ESQL sources including indices, aliases, data streams, and integrations.
 * @param core The core start contract to make HTTP requests and access application capabilities.
 * @param getLicense An optional function to retrieve the current license information.
 * @returns A promise that resolves to an array of ESQLSourceResult objects.
 */
const getESQLSources = async (core, getLicense) => {
  var _ccrFeature$isAvailab;
  const ls = await (getLicense === null || getLicense === void 0 ? void 0 : getLicense());
  const ccrFeature = ls === null || ls === void 0 ? void 0 : ls.getFeature('ccr');
  const areRemoteIndicesAvailable = (_ccrFeature$isAvailab = ccrFeature === null || ccrFeature === void 0 ? void 0 : ccrFeature.isAvailable) !== null && _ccrFeature$isAvailab !== void 0 ? _ccrFeature$isAvailab : false;
  const [allIndices, integrations] = await Promise.all([getIndicesList(core, areRemoteIndicesAvailable), getIntegrations(core)]);
  return [...allIndices, ...integrations];
};
exports.getESQLSources = getESQLSources;