"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findSubquery = findSubquery;
var _ast = require("../../ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Finds the innermost subquery containing the cursor position and determines if the query contains subqueries.
 *
 * Example: FROM a, (FROM b, (FROM c | WHERE |))
 *                                        ↑ cursor
 * Returns: { subQuery: ESQLAstQueryExpression for "FROM c | WHERE", queryContainsSubqueries: true }
 */
function findSubquery(queryAst, offset) {
  let subQuery = null;
  let queryContainsSubqueries = false;

  // Check if query contains subqueries while walking the AST
  const allCommands = _ast.Walker.commands(queryAst);
  const fromCommands = allCommands.filter(({
    name
  }) => name.toLowerCase() === 'from');
  queryContainsSubqueries = fromCommands.some(cmd => cmd.args.some(arg => (0, _ast.isSubQuery)(arg)));
  _ast.Walker.walk(queryAst, {
    visitParens: (node, parent) => {
      const isForkBranch = (parent === null || parent === void 0 ? void 0 : parent.type) === 'command' && parent.name === 'fork';
      if ((0, _ast.isSubQuery)(node) && (0, _ast.within)(offset, node) && !isForkBranch) {
        subQuery = node.child;
      }
    }
  });
  return {
    subQuery,
    queryContainsSubqueries
  };
}