"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCursorContext = getCursorContext;
var _ast = require("../../ast");
var _utils = require("../../commands/definitions/utils");
var _ast2 = require("../../commands/definitions/utils/ast");
var _subqueries_helpers = require("./subqueries_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Given a ES|QL query string, its AST and the cursor position,
 * it returns the type of context for the position ("list", "function", "option", "setting", "expression", "newCommand")
 * plus the whole hierarchy of nodes (command, option, setting and actual position node) context.
 *
 * Type details:
 * * "list": the cursor is inside a "in" list of values (i.e. `a in (1, 2, <here>)`)
 * * "function": the cursor is inside a function call (i.e. `fn(<here>)`)
 * * "expression": the cursor is inside a command expression (i.e. `command ... <here>` or `command a = ... <here>`)
 * * "newCommand": the cursor is at the beginning of a new command (i.e. `command1 | command2 | <here>`)
 */
function getCursorContext(queryString, queryAst, offset) {
  const {
    subQuery,
    queryContainsSubqueries
  } = (0, _subqueries_helpers.findSubquery)(queryAst, offset);
  const astForContext = subQuery !== null && subQuery !== void 0 ? subQuery : queryAst;
  const isCursorInSubquery = subQuery !== null;
  let inComment = false;
  _ast.Walker.visitComments(queryAst, node => {
    // if the cursor (offset) is within the range of a comment node
    if ((0, _ast.within)(offset, node)) {
      inComment = true;
      // or if the cursor (offset) is right after a single-line comment (which means there was no newline)
    } else if (node.subtype === 'single-line' && node.location && offset === node.location.max + 1) {
      inComment = true;
    }
  });
  if (inComment) {
    return {
      type: 'comment',
      isCursorInSubquery,
      queryContainsSubqueries,
      astForContext
    };
  }
  const {
    command,
    option,
    node,
    containingFunction
  } = (0, _ast2.findAstPosition)(astForContext, offset);
  if (!command || queryString.length <= offset && (0, _utils.pipePrecedesCurrentWord)(queryString) && command.location.max < queryString.length) {
    //   // ... | <here>
    return {
      type: 'newCommand',
      command: undefined,
      node,
      option,
      containingFunction,
      isCursorInSubquery,
      queryContainsSubqueries,
      astForContext
    };
  }

  // command a ... <here> OR command a = ... <here>
  return {
    type: 'expression',
    command,
    containingFunction,
    option,
    node,
    isCursorInSubquery,
    queryContainsSubqueries,
    astForContext
  };
}