"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.byCompleteItem = void 0;
exports.checkAggExistence = checkAggExistence;
exports.checkFunctionContent = checkFunctionContent;
exports.whereCompleteItem = exports.rightAfterColumn = exports.getPosition = exports.getCommaAndPipe = void 0;
var _helpers = require("../../definitions/utils/autocomplete/helpers");
var _complete_items = require("../complete_items");
var _is = require("../../../ast/is");
var _walker = require("../../../ast/walker");
var _functions = require("../../definitions/utils/functions");
var _types = require("../../definitions/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Position of the caret in the sort command:
*
* ```
* STATS [column1 =] expression1[, ..., [columnN =] expressionN] [BY [column1 =] grouping_expression1[, ..., grouping_expressionN]]
        |           |          |                                    |           |                   |
        |           |          expression_complete                  |           |                   grouping_expression_complete
        |           expression_after_assignment                     |           grouping_expression_after_assignment
        expression_without_assignment                               grouping_expression_without_assignment

* ```
*/

const ENDS_WITH_COMMA_AND_WHITESPACE_REGEX = /,\s*$/;
const getPosition = (command, innerText) => {
  const lastCommandArg = command.args[command.args.length - 1];
  if ((0, _is.isOptionNode)(lastCommandArg) && lastCommandArg.name === 'by') {
    // in the BY clause

    const lastOptionArg = lastCommandArg.args[lastCommandArg.args.length - 1];
    if ((0, _is.isAssignment)(lastOptionArg) && !ENDS_WITH_COMMA_AND_WHITESPACE_REGEX.test(innerText)) {
      return 'grouping_expression_after_assignment';
    }
    return 'grouping_expression_without_assignment';
  }
  if ((0, _is.isAssignment)(lastCommandArg) && !ENDS_WITH_COMMA_AND_WHITESPACE_REGEX.test(innerText)) {
    return 'expression_after_assignment';
  }
  if ((0, _is.isFunctionExpression)(lastCommandArg) && lastCommandArg.name === 'where' && !ENDS_WITH_COMMA_AND_WHITESPACE_REGEX.test(innerText)) {
    return 'after_where';
  }
  return 'expression_without_assignment';
};
exports.getPosition = getPosition;
const byCompleteItem = exports.byCompleteItem = (0, _helpers.withAutoSuggest)({
  label: 'BY',
  text: 'BY ',
  kind: 'Reference',
  detail: 'By',
  sortText: '1'
});
const whereCompleteItem = exports.whereCompleteItem = (0, _helpers.withAutoSuggest)({
  label: 'WHERE',
  text: 'WHERE ',
  kind: 'Reference',
  detail: 'Where',
  sortText: '1'
});
function isAggregation(arg) {
  var _getFunctionDefinitio;
  return (0, _is.isFunctionExpression)(arg) && ((_getFunctionDefinitio = (0, _functions.getFunctionDefinition)(arg.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.AGG;
}
function isNotAnAggregation(arg) {
  var _getFunctionDefinitio2;
  return (0, _is.isFunctionExpression)(arg) && ((_getFunctionDefinitio2 = (0, _functions.getFunctionDefinition)(arg.name)) === null || _getFunctionDefinitio2 === void 0 ? void 0 : _getFunctionDefinitio2.type) !== _types.FunctionDefinitionTypes.AGG;
}
const isFunctionOperatorParam = fn => !!fn.operator && (0, _is.isParamLiteral)(fn.operator);
function checkAggExistence(arg) {
  if ((0, _is.isWhereExpression)(arg)) {
    return checkAggExistence(arg.args[0]);
  }
  if ((0, _is.isFieldExpression)(arg)) {
    const agg = arg.args[1];
    const firstFunction = _walker.Walker.match(agg, {
      type: 'function'
    });
    if (!firstFunction) {
      return false;
    }
    return checkAggExistence(firstFunction);
  }

  // TODO the grouping function check may not
  // hold true for all future cases
  if (isAggregation(arg) || isFunctionOperatorParam(arg)) {
    return true;
  }
  if (isNotAnAggregation(arg)) {
    return arg.args.filter(_is.isFunctionExpression).some(checkAggExistence);
  }
  return false;
}

// now check that:
// * the agg function is at root level
// * or if it's a operators function, then all operands are agg functions or literals
// * or if it's a eval function then all arguments are agg functions or literals
// * or if a named param is used
function checkFunctionContent(arg) {
  // TODO the grouping function check may not
  // hold true for all future cases
  if (isAggregation(arg) || isFunctionOperatorParam(arg)) {
    return true;
  }
  return arg.args.every(subArg => {
    // Differentiate between array and non-array arguments
    if (Array.isArray(subArg)) {
      return subArg.every(item => checkFunctionContent(item));
    }
    return (0, _is.isLiteral)(subArg) || isAggregation(subArg) || (isNotAnAggregation(subArg) ? checkFunctionContent(subArg) : false);
  });
}
const rightAfterColumn = (innerText, expressionRoot, columnExists) => {
  if (!expressionRoot) return false;
  let col;
  _walker.Walker.walk(expressionRoot, {
    visitColumn(node) {
      if (node.location.max === innerText.length - 1) col = node;
    }
  });
  return (0, _is.isColumn)(col) && columnExists(col.parts.join('.'));
};
exports.rightAfterColumn = rightAfterColumn;
const getCommaAndPipe = (innerText, expressionRoot, columnExists) => {
  const pipeSuggestion = {
    ..._complete_items.pipeCompleteItem
  };
  const commaSuggestion = (0, _helpers.withAutoSuggest)({
    ..._complete_items.commaCompleteItem,
    text: ', '
  });

  // does the query end with whitespace?
  if (/\s$/.test(innerText)) {
    // if so, comma needs to be sent back a column to replace the trailing space
    commaSuggestion.rangeToReplace = {
      start: innerText.length - 1,
      end: innerText.length
    };
  }
  // special case: cursor right after a column name
  else if ((0, _is.isColumn)(expressionRoot) && rightAfterColumn(innerText, expressionRoot, columnExists)) {
    const {
      fragment,
      rangeToReplace
    } = (0, _helpers.getFragmentData)(innerText);
    pipeSuggestion.filterText = fragment;
    pipeSuggestion.text = fragment + ' ' + pipeSuggestion.text;
    pipeSuggestion.rangeToReplace = rangeToReplace;
    commaSuggestion.filterText = fragment;
    commaSuggestion.text = fragment + commaSuggestion.text;
    commaSuggestion.rangeToReplace = rangeToReplace;
  }
  return [pipeSuggestion, commaSuggestion];
};
exports.getCommaAndPipe = getCommaAndPipe;