"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _esqlTypes = require("@kbn/esql-types");
var _complete_items = require("../complete_items");
var _sources = require("../../definitions/utils/sources");
var _metadata = require("../options/metadata");
var _recommended_queries = require("../options/recommended_queries");
var _helpers = require("../../definitions/utils/autocomplete/helpers");
var _shared = require("../../definitions/utils/shared");
var _is = require("../../../ast/is");
var _ = require("../../../..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SOURCE_TYPE_INDEX = 'index';
const METADATA_KEYWORD = 'METADATA';
const EMPTY_EXTENSIONS = {
  recommendedFields: [],
  recommendedQueries: []
};
const PIPE_SORT_TEXT = '0';
async function autocomplete(query, command, callbacks, context, cursorPosition) {
  const innerText = query.substring(0, cursorPosition);
  if ((0, _helpers.withinQuotes)(innerText) || !(callbacks !== null && callbacks !== void 0 && callbacks.getByType)) {
    return [];
  }
  return handleFromAutocomplete(query, command, callbacks, context, cursorPosition);
}

/**
 * Routes to appropriate suggestion scenario based on cursor position and indexes.
 */
async function handleFromAutocomplete(query, command, callbacks, context, cursorPosition) {
  const cursorPos = cursorPosition !== null && cursorPosition !== void 0 ? cursorPosition : query.length;
  const innerText = query.substring(0, cursorPos);

  // Cursor before FROM keyword
  if (command.location.min > cursorPos) {
    var _context$sources;
    return (0, _sources.getSourceSuggestions)((_context$sources = context === null || context === void 0 ? void 0 : context.sources) !== null && _context$sources !== void 0 ? _context$sources : [], [], innerText);
  }

  // Extract text relative to command start (critical for subqueries)
  // Use commandText for pattern matching (e.g., /METADATA\s+$/, /\s$/) because these
  // checks need to operate on the current command only, not the entire query
  const commandText = query.substring(command.location.min, cursorPos);

  // METADATA suggestions - uses commandText for regex pattern matching
  const metadataSuggestions = await (0, _metadata.getMetadataSuggestions)(command, commandText);
  if (metadataSuggestions) {
    return metadataSuggestions;
  }
  const indexes = (0, _sources.getSourcesFromCommands)([command], SOURCE_TYPE_INDEX);

  // Check if there are any sources (including subqueries)
  const hasAnySources = command.args.some(arg => !Array.isArray(arg) && ((0, _is.isSource)(arg) || (0, _is.isSubQuery)(arg)));

  // Case 1: FROM | (no sources yet)
  if (!hasAnySources) {
    // Use innerText for absolute positions in rangeToReplace
    return suggestInitialSources(context, innerText);
  }

  // Case 2: FROM index | (after space, suggest next actions)
  if (/\s$/.test(commandText) && !(0, _shared.isRestartingExpression)(commandText)) {
    return suggestNextActions(context, callbacks);
  }

  // Case 3: FROM in|, FROM index, | (typing or adding more indexes)
  // Use innerText for absolute positions in rangeToReplace
  return suggestAdditionalSources(innerText, context, callbacks, indexes);
}

/**
 * Case 1: No indexes yet - suggest available sources and subquery.
 */
function suggestInitialSources(context, innerText) {
  var _context$sources2;
  let sources = (_context$sources2 = context === null || context === void 0 ? void 0 : context.sources) !== null && _context$sources2 !== void 0 ? _context$sources2 : [];
  if (context !== null && context !== void 0 && context.isCursorInSubquery) {
    sources = sources.filter(source => source.type !== _esqlTypes.SOURCES_TYPES.TIMESERIES);
  }
  const suggestions = (0, _sources.getSourceSuggestions)(sources, [], innerText);
  if (shouldSuggestSubquery(context)) {
    suggestions.push(_complete_items.subqueryCompleteItem);
  }
  return suggestions;
}

/**
 * Case 2: After space - suggest pipe, comma, metadata, and recommended queries.
 */
async function suggestNextActions(context, callbacks) {
  var _context$editorExtens;
  const suggestions = [{
    ..._complete_items.pipeCompleteItem,
    sortText: PIPE_SORT_TEXT
  }, _complete_items.commaCompleteItem, _metadata.metadataSuggestion];
  const recommendedQueries = await (0, _recommended_queries.getRecommendedQueriesSuggestions)((_context$editorExtens = context === null || context === void 0 ? void 0 : context.editorExtensions) !== null && _context$editorExtens !== void 0 ? _context$editorExtens : EMPTY_EXTENSIONS, callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType);
  return [...suggestions, ...recommendedQueries];
}

/**
 * Case 3: Typing or adding more indexes - suggest additional sources with metadata check.
 */
async function suggestAdditionalSources(innerText, context, callbacks, indexes) {
  var _context$sources3, _context$editorExtens2;
  const lastIndex = indexes[indexes.length - 1];
  const isTypingIndexName = (lastIndex === null || lastIndex === void 0 ? void 0 : lastIndex.name) && innerText.endsWith(lastIndex.name);

  // Check for METADATA overlap (only when not typing index name)
  if (!isTypingIndexName && (0, _shared.getOverlapRange)(innerText, METADATA_KEYWORD)) {
    return [_metadata.metadataSuggestion];
  }
  let sources = (_context$sources3 = context === null || context === void 0 ? void 0 : context.sources) !== null && _context$sources3 !== void 0 ? _context$sources3 : [];
  if (context !== null && context !== void 0 && context.isCursorInSubquery) {
    sources = sources.filter(source => source.type !== _esqlTypes.SOURCES_TYPES.TIMESERIES);
  }
  const recommendedQueries = await (0, _recommended_queries.getRecommendedQueriesSuggestions)((_context$editorExtens2 = context === null || context === void 0 ? void 0 : context.editorExtensions) !== null && _context$editorExtens2 !== void 0 ? _context$editorExtens2 : EMPTY_EXTENSIONS, callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType);
  const suggestions = await (0, _sources.additionalSourcesSuggestions)(innerText, sources, indexes.map(({
    name
  }) => name), recommendedQueries);
  if ((0, _shared.isRestartingExpression)(innerText) && shouldSuggestSubquery(context)) {
    suggestions.push(_complete_items.subqueryCompleteItem);
  }
  return suggestions;
}
function shouldSuggestSubquery(context) {
  var _fromCommand$metadata, _fromCommand$metadata2;
  if (context !== null && context !== void 0 && context.isCursorInSubquery) {
    return false;
  }
  const fromCommand = _.esqlCommandRegistry.getCommandByName('from');
  return (_fromCommand$metadata = fromCommand === null || fromCommand === void 0 ? void 0 : (_fromCommand$metadata2 = fromCommand.metadata) === null || _fromCommand$metadata2 === void 0 ? void 0 : _fromCommand$metadata2.subquerySupport) !== null && _fromCommand$metadata !== void 0 ? _fromCommand$metadata : true;
}