"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestInFunction = suggestInFunction;
var _suggestion_engine = require("../suggestion_engine");
var _functions = require("../../../functions");
var _signature_analyzer = require("../signature_analyzer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** Matches comma followed by optional whitespace at end of text */
const STARTING_NEW_PARAM_REGEX = /,\s*$/;

/** Suggests completions when cursor is inside a function call (e.g., CONCAT(field1, /)) */
async function suggestInFunction(ctx) {
  const {
    expressionRoot,
    context,
    options,
    innerText,
    query,
    command,
    cursorPosition,
    location,
    callbacks
  } = ctx;
  const functionExpression = expressionRoot;
  const functionDefinition = (0, _functions.getFunctionDefinition)(functionExpression.name);
  if (!functionDefinition || !context) {
    return [];
  }
  const analyzer = _signature_analyzer.SignatureAnalyzer.fromNode(functionExpression, context, functionDefinition);
  if (!analyzer) {
    return [];
  }
  const paramContext = buildInFunctionParameterContext(analyzer, functionExpression.name, functionDefinition, options.functionParameterContext);
  const targetExpression = determineTargetExpression(functionExpression, innerText);
  const {
    suggestions
  } = await (0, _suggestion_engine.suggestForExpression)({
    query,
    command,
    cursorPosition,
    location,
    context,
    callbacks,
    expressionRoot: targetExpression,
    options: {
      ...options,
      functionParameterContext: paramContext
    }
  });
  return suggestions;
}

/** Builds function parameter context, adding current function to ignore list */
function buildInFunctionParameterContext(analyzer, functionName, functionDefinition, existingContext) {
  const existingIgnored = (existingContext === null || existingContext === void 0 ? void 0 : existingContext.functionsToIgnore) || [];
  const functionsToIgnore = existingIgnored.includes(functionName) ? existingIgnored : [...existingIgnored, functionName];
  return {
    paramDefinitions: analyzer.getCompatibleParamDefs(),
    functionsToIgnore,
    hasMoreMandatoryArgs: analyzer.getHasMoreMandatoryArgs(),
    functionDefinition,
    firstArgumentType: analyzer.getFirstArgumentType(),
    firstValueType: analyzer.getFirstValueType(),
    currentParameterIndex: analyzer.getCurrentParameterIndex(),
    validSignatures: analyzer.getValidSignatures()
  };
}

/** Determines which expression to use as target for recursive suggestion */
function determineTargetExpression(functionExpression, innerText) {
  const {
    args
  } = functionExpression;
  const startingNewParam = STARTING_NEW_PARAM_REGEX.test(innerText);
  const firstArgEmpty = isFirstArgumentEmpty(args, innerText);
  if (startingNewParam || firstArgEmpty) {
    return undefined;
  }
  const lastArg = args[args.length - 1];
  return Array.isArray(lastArg) ? lastArg[0] : lastArg;
}

/** Checks if cursor is immediately after opening parenthesis with no argument */
function isFirstArgumentEmpty(args, innerText) {
  if (args.length === 0 || !args[0]) {
    return false;
  }
  const firstArgIsEmpty = Array.isArray(args[0]) ? args[0].length === 0 : !args[0];
  return firstArgIsEmpty && innerText.trimEnd().endsWith('(');
}