"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPosition = getPosition;
var _is = require("../../../../../ast/is");
var _location = require("../../../../../ast/location");
var _utils = require("./utils");
var _functions = require("../../functions");
var _expressions = require("../../expressions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** Matches " not" at end of string (case insensitive) */
const NOT_PATTERN = / not$/i;
/** Matches all regex special characters: . * + ? ^ $ { } ( ) | [ ] \ */
const REGEX_SPECIAL_CHARS = /[.*+?^${}()|[\]\\]/g;

/** Determines the position of the cursor within an expression */
function getPosition(innerText, expressionRoot, columns) {
  // Handle NOT keyword first (must be before empty check for proper precedence)
  const endsWithNot = NOT_PATTERN.test(innerText.trimEnd());
  const isNullCheck = expressionRoot && (0, _is.isFunctionExpression)(expressionRoot) && (0, _utils.isNullCheckOperator)(expressionRoot.name);
  if (endsWithNot && !isNullCheck) {
    return 'after_not';
  }
  if (!expressionRoot) {
    return 'empty_expression';
  }
  if ((0, _is.isColumn)(expressionRoot)) {
    const escapedColumn = expressionRoot.parts.join('\\.').replace(REGEX_SPECIAL_CHARS, '\\$&');
    const endsWithColumnName = new RegExp(`${escapedColumn}$`).test(innerText);

    // If cursor is after column but text continues, suggest operators
    if (!endsWithColumnName) {
      return 'after_complete';
    }
  }

  // Function expression (operators or variadic functions like CONCAT)
  if ((0, _is.isFunctionExpression)(expressionRoot)) {
    if (expressionRoot.subtype === 'variadic-call') {
      const cursorIsInside = (0, _location.within)(innerText.length, expressionRoot);
      return cursorIsInside ? 'in_function' : 'after_complete';
    }

    // Postfix unary operators (IS NULL, IS NOT NULL) are complete when not marked incomplete
    // Incomplete means partial typing like "IS N" - should be handled by pre-pass
    if (expressionRoot.subtype === 'postfix-unary-expression' && !expressionRoot.incomplete) {
      return 'after_complete';
    }

    // Binary operators (e.g., "field = |", "field IN |")
    // Check if operator is complete (has both operands)
    if (expressionRoot.subtype === 'binary-expression' && columns) {
      const {
        complete
      } = (0, _functions.checkFunctionInvocationComplete)(expressionRoot, expr => (0, _expressions.getExpressionType)(expr, columns));
      return complete ? 'after_complete' : 'after_operator';
    }

    // Fallback: if no columns available, use old behavior
    return 'after_operator';
  }
  if ((0, _is.isLiteral)(expressionRoot)) {
    return 'after_complete';
  }
  return 'empty_expression';
}