"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSyntheticLikeOperatorNode = createSyntheticLikeOperatorNode;
exports.createSyntheticListOperatorNode = createSyntheticListOperatorNode;
exports.detectIn = detectIn;
exports.detectLike = detectLike;
exports.detectNullCheck = detectNullCheck;
var _utils = require("../utils");
var _builder = require("../../../../../../../ast/builder");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Regex to extract field name before operator: match[1] = fieldName
// Matches with or without opening parenthesis
const FIELD_BEFORE_IN_REGEX = /([\w.]+)\s+(?:not\s+)?in\s*\(?\s*$/i;
const FIELD_BEFORE_LIKE_REGEX = /([\w.]+)\s+(?:not\s+)?(?:r)?like\s*\(?\s*$/i;

/**
 * Creates a synthetic infix operator node (IN, LIKE, RLIKE, etc.).
 * Used when cursor is right after operator: "field IN ", "field IN(", "field LIKE ".
 * If innerText ends with "(", creates a list node instead of placeholder.
 */
function createSyntheticInfixOperatorNode(operatorName, innerText, fieldPattern, leftOperand) {
  const textLength = innerText.length;
  const hasOpenParen = /\(\s*$/.test(innerText);
  const right = hasOpenParen ? createEmptyListNode(textLength) : createPlaceholderNode(textLength);
  const left = leftOperand !== null && leftOperand !== void 0 ? leftOperand : extractFieldFromText(innerText, fieldPattern);
  return {
    type: 'function',
    name: operatorName,
    subtype: 'binary-expression',
    args: [left !== null && left !== void 0 ? left : createPlaceholderNode(0), right],
    incomplete: true,
    location: {
      min: textLength,
      max: textLength
    },
    text: operatorName
  };
}
function createSyntheticListOperatorNode(operatorName, innerText, leftOperand) {
  return createSyntheticInfixOperatorNode(operatorName, innerText, FIELD_BEFORE_IN_REGEX, leftOperand);
}
function createSyntheticLikeOperatorNode(operatorName, innerText, leftOperand) {
  return createSyntheticInfixOperatorNode(operatorName, innerText, FIELD_BEFORE_LIKE_REGEX, leftOperand);
}
function createPlaceholderNode(textLength) {
  return {
    type: 'unknown',
    name: '',
    text: '',
    location: {
      min: textLength,
      max: textLength
    },
    incomplete: true
  };
}
function createEmptyListNode(textLength) {
  return _builder.Builder.expression.list.tuple({
    text: '()',
    location: {
      min: textLength,
      max: textLength
    },
    incomplete: true
  }, {
    location: {
      min: textLength,
      max: textLength
    },
    text: '()',
    incomplete: true
  });
}
function extractFieldFromText(innerText, pattern) {
  const match = innerText.match(pattern);
  if (match !== null && match !== void 0 && match[1]) {
    return _builder.Builder.expression.column(match[1]);
  }
  return undefined;
}

/**
 * Detects partial IS NULL / IS NOT NULL operators.
 * Examples: "field IS ", "field IS N", "field IS NOT ", "field IS NOT N"
 */
function detectNullCheck(innerText) {
  if (!(0, _utils.endsWithIsOrIsNotToken)(innerText)) {
    return null;
  }

  // Check if it contains NOT to determine which operator
  const containsNot = _utils.IS_NOT_REGEX.test(innerText);
  return {
    operatorName: containsNot ? 'is not null' : 'is null',
    textBeforeCursor: innerText
  };
}

/**
 * Detects partial LIKE / RLIKE / NOT LIKE / NOT RLIKE operators.
 * Examples: "field LIKE ", "field RLIKE ", "field NOT LIKE ", "field NOT RLIKE "
 */
function detectLike(innerText) {
  if (!(0, _utils.endsWithLikeOrRlikeToken)(innerText)) {
    return null;
  }
  const match = innerText.match(_utils.LIKE_OPERATOR_REGEX);
  if (!match) {
    return null;
  }

  // Normalize: lowercase, trim, collapse multiple spaces
  const operatorName = match[0].toLowerCase().trim().replace(/\s+/g, ' ');
  return {
    operatorName,
    textBeforeCursor: innerText
  };
}

/**
 * Detects partial IN / NOT IN operators.
 * Examples: "field IN ", "field IN(", "field NOT IN ", "field NOT IN("
 */
function detectIn(innerText) {
  // Don't clean the text - we want to preserve parentheses to NOT match them
  if (!(0, _utils.endsWithInOrNotInToken)(innerText)) {
    return null;
  }
  const isNotIn = _utils.NOT_IN_REGEX.test(innerText);
  return {
    operatorName: isNotIn ? 'not in' : 'in',
    textBeforeCursor: innerText
  };
}