"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.remove = exports.insert = exports.append = void 0;
var _is = require("../../../../is");
var _visitor = require("../../../../visitor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const insert = (command, expression, index = -1) => {
  if (expression.type === 'option') {
    command.args.push(expression);
    return command.args.length - 1;
  }
  let mainArgumentCount = command.args.findIndex(arg => (0, _is.isOptionNode)(arg));
  if (mainArgumentCount < 0) {
    mainArgumentCount = command.args.length;
  }
  if (index === -1) {
    index = mainArgumentCount;
  }
  if (index > mainArgumentCount) {
    index = mainArgumentCount;
  }
  command.args.splice(index, 0, expression);
  return mainArgumentCount + 1;
};
exports.insert = insert;
const append = (command, expression) => {
  return insert(command, expression, -1);
};

/**
 * Searches all command arguments in the query AST node and removes the node
 * from the command's arguments list. This function recursively searches through
 * subqueries as well.
 *
 * @param ast The root AST node to search for command arguments.
 * @param node The argument AST node to remove.
 * @returns Returns the command that the argument was removed from, if any.
 */
exports.append = append;
const remove = (ast, node) => {
  return new _visitor.Visitor().on('visitCommand', ctx => {
    const args = ctx.node.args;
    const length = args.length;
    for (let i = 0; i < length; i++) {
      const arg = args[i];
      if (arg === node) {
        args.splice(i, 1);
        return ctx.node;
      }
      if (!Array.isArray(arg) && (0, _is.isSubQuery)(arg)) {
        const found = remove(arg.child, node);
        if (found) {
          return found;
        }
      }
    }
    return undefined;
  }).on('visitQuery', ctx => {
    for (const cmd of ctx.visitCommands()) {
      if (cmd) {
        return cmd;
      }
    }
    return undefined;
  }).visitQuery(ast);
};
exports.remove = remove;