"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformToBaseAlertDocument = exports.transformToAlertDocuments = exports.generateAttackDiscoveryAlertHash = void 0;
var _crypto = require("crypto");
var _ecs = require("@elastic/ecs");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _fp = require("lodash/fp");
var _get_alert_risk_score = require("./get_alert_risk_score");
var _field_names = require("../../../schedules/fields/field_names");
var _get_alert_url = require("./get_alert_url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const generateAttackDiscoveryAlertHash = ({
  attackDiscovery,
  connectorId,
  ownerId,
  replacements,
  spaceId
}) => {
  /**
   * We generate an attack discovery alert hash to uniquely identify the attack.
   * The hash is a combination of:
   * - `alertsId` - the set of the detection alerts that led to the attack
   * - `connectorId` - to separate same attacks generated by different LLM connectors
   * - `ownerId` - to separate same attacks generated by different users (ad hoc alerts) or different schedules
   *               We store ad hoc attacks for all users within the same index and that is why we need this separation.
   * - `spaceId` - to separate attacks on a space basis
   */
  const alertIds = attackDiscovery.alertIds;
  const originalAlertIds = (0, _elasticAssistantCommon.getOriginalAlertIds)({
    alertIds,
    replacements
  });
  return (0, _crypto.createHash)('sha256').update([...originalAlertIds].sort().join()).update(connectorId).update(ownerId).update(spaceId).digest('hex');
};
exports.generateAttackDiscoveryAlertHash = generateAttackDiscoveryAlertHash;
const transformToBaseAlertDocument = ({
  alertDocId,
  alertInstanceId,
  attackDiscovery,
  alertsParams,
  publicBaseUrl,
  spaceId
}) => {
  const {
    alertsContextCount,
    anonymizedAlerts,
    apiConfig,
    connectorName,
    replacements
  } = alertsParams;
  const {
    alertIds,
    entitySummaryMarkdown,
    detailsMarkdown,
    mitreAttackTactics,
    summaryMarkdown,
    title
  } = attackDiscovery;
  const baseAlertDocument = {
    // Alert base fields
    [_ruleDataUtils.ECS_VERSION]: _ecs.EcsVersion,
    [_ruleDataUtils.ALERT_INSTANCE_ID]: alertInstanceId,
    [_ruleDataUtils.ALERT_UUID]: alertDocId,
    // IMPORTANT: the document _id should be the same as this field when it's bulk inserted
    [_field_names.ALERT_RISK_SCORE]: (0, _get_alert_risk_score.getAlertRiskScore)({
      alertIds,
      anonymizedAlerts
    }),
    [_ruleDataUtils.ALERT_URL]: (0, _get_alert_url.getAlertUrl)({
      alertDocId,
      basePath: publicBaseUrl,
      spaceId
    }),
    // Attack discovery fields
    [_field_names.ALERT_ATTACK_DISCOVERY_ALERT_IDS]: alertIds,
    [_field_names.ALERT_ATTACK_DISCOVERY_ALERTS_CONTEXT_COUNT]: alertsContextCount,
    [_field_names.ALERT_ATTACK_DISCOVERY_API_CONFIG]: {
      action_type_id: apiConfig.actionTypeId,
      connector_id: apiConfig.connectorId,
      model: apiConfig.model,
      name: connectorName,
      provider: apiConfig.provider
    },
    [_field_names.ALERT_ATTACK_DISCOVERY_DETAILS_MARKDOWN]: detailsMarkdown,
    [_field_names.ALERT_ATTACK_DISCOVERY_DETAILS_MARKDOWN_WITH_REPLACEMENTS]: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
      messageContent: detailsMarkdown,
      replacements
    }),
    [_field_names.ALERT_ATTACK_DISCOVERY_ENTITY_SUMMARY_MARKDOWN]: entitySummaryMarkdown,
    [_field_names.ALERT_ATTACK_DISCOVERY_ENTITY_SUMMARY_MARKDOWN_WITH_REPLACEMENTS]: entitySummaryMarkdown != null ? (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
      messageContent: entitySummaryMarkdown,
      replacements
    }) : undefined,
    [_field_names.ALERT_ATTACK_DISCOVERY_MITRE_ATTACK_TACTICS]: mitreAttackTactics,
    [_field_names.ALERT_ATTACK_DISCOVERY_REPLACEMENTS]: !(0, _fp.isEmpty)(replacements) ? Object.entries(replacements).map(([uuid, value]) => ({
      uuid,
      value
    })) : undefined,
    [_field_names.ALERT_ATTACK_DISCOVERY_SUMMARY_MARKDOWN]: summaryMarkdown,
    [_field_names.ALERT_ATTACK_DISCOVERY_SUMMARY_MARKDOWN_WITH_REPLACEMENTS]: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
      messageContent: summaryMarkdown,
      replacements
    }),
    [_field_names.ALERT_ATTACK_DISCOVERY_TITLE]: title,
    [_field_names.ALERT_ATTACK_DISCOVERY_TITLE_WITH_REPLACEMENTS]: (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
      messageContent: title,
      replacements
    }),
    /**
     * This field is shared with security solution alerts.
     * We want both attacks and alerts to have this field so
     * we can filter and group them in the security attacks
     * page using both the attacks and the alerts indexes.
     *
     * @see https://github.com/elastic/kibana/issues/232341
     */
    [_field_names.ALERT_ATTACK_IDS]: [alertDocId]
  };
  return baseAlertDocument;
};
exports.transformToBaseAlertDocument = transformToBaseAlertDocument;
const transformToAlertDocuments = ({
  authenticatedUser,
  createAttackDiscoveryAlertsParams,
  now,
  spaceId
}) => {
  const {
    attackDiscoveries,
    generationUuid,
    ...restParams
  } = createAttackDiscoveryAlertsParams;
  return attackDiscoveries.map(attackDiscovery => {
    var _authenticatedUser$us;
    const alertHash = generateAttackDiscoveryAlertHash({
      attackDiscovery,
      connectorId: restParams.apiConfig.connectorId,
      ownerId: (_authenticatedUser$us = authenticatedUser.username) !== null && _authenticatedUser$us !== void 0 ? _authenticatedUser$us : authenticatedUser.profile_uid,
      replacements: restParams.replacements,
      spaceId
    });
    const baseAlertDocument = transformToBaseAlertDocument({
      alertDocId: alertHash,
      alertInstanceId: alertHash,
      attackDiscovery,
      alertsParams: restParams,
      spaceId
    });
    return {
      ...baseAlertDocument,
      '@timestamp': now.toISOString(),
      [_ruleDataUtils.ALERT_START]: now.toISOString(),
      [_field_names.ALERT_ATTACK_DISCOVERY_USER_ID]: authenticatedUser.profile_uid,
      [_field_names.ALERT_ATTACK_DISCOVERY_USER_NAME]: authenticatedUser.username,
      [_field_names.ALERT_ATTACK_DISCOVERY_USERS]: [{
        id: authenticatedUser.profile_uid,
        name: authenticatedUser.username
      }],
      [_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]: generationUuid,
      [_ruleDataUtils.ALERT_RULE_CATEGORY]: 'Attack discovery ad hoc (placeholder rule category)',
      [_ruleDataUtils.ALERT_RULE_CONSUMER]: 'siem',
      [_ruleDataUtils.ALERT_RULE_NAME]: 'Attack discovery ad hoc (placeholder rule name)',
      [_ruleDataUtils.ALERT_RULE_PRODUCER]: 'siem',
      [_ruleDataUtils.ALERT_RULE_REVISION]: 1,
      [_ruleDataUtils.ALERT_RULE_TYPE_ID]: _elasticAssistantCommon.ATTACK_DISCOVERY_AD_HOC_RULE_TYPE_ID,
      // sentinel value
      [_ruleDataUtils.ALERT_RULE_UUID]: _elasticAssistantCommon.ATTACK_DISCOVERY_AD_HOC_RULE_ID,
      // sentinel value
      [_ruleDataUtils.ALERT_STATUS]: 'active',
      [_ruleDataUtils.ALERT_WORKFLOW_STATUS]: 'open',
      [_ruleDataUtils.EVENT_KIND]: 'signal',
      [_ruleDataUtils.SPACE_IDS]: [spaceId]
    };
  });
};
exports.transformToAlertDocuments = transformToAlertDocuments;