"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isNestedFieldParent = isNestedFieldParent;
var _fp = require("lodash/fp");
var _esQuery = require("@kbn/es-query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * This function checks if the given field in a given data view is a nested field's parent.
 * Discover doesn't flatten arrays of objects, so for documents with an `object` or `nested` field that
 * contains an array, Discover will only detect the top level root field. We want to detect when those
 * root fields are `nested` so that we can display the proper icon and label. However, those root
 * `nested` fields are not a part of the data view. Their children are though, and contain nested path
 * info. So to detect nested fields we look through the data view for nested children
 * whose path begins with the current field. There are edge cases where
 * this could incorrectly identify a plain `object` field as `nested`. Say we had the following document
 * where `foo` is a plain object field and `bar` is a nested field.
 * {
 *   "foo": [
 *   {
 *     "bar": [
 *       {
 *         "baz": "qux"
 *       }
 *     ]
 *   },
 *   {
 *     "bar": [
 *       {
 *         "baz": "qux"
 *       }
 *     ]
 *   }
 * ]
 * }
 * The following code will search for `foo`, find it at the beginning of the path to the nested child field
 * `foo.bar.baz` and incorrectly mark `foo` as nested. Any time we're searching for the name of a plain object
 * field that happens to match a segment of a nested path, we'll get a false positive.
 * We're aware of this issue and we'll have to live with
 * it in the short term. The long term fix will be to add info about the `nested` and `object` root fields
 * to the data view, but that has its own complications which you can read more about in the following
 * issue: https://github.com/elastic/kibana/issues/54957
 */
function isNestedFieldParent(fieldName, dataView) {
  const nestedRootRegex = new RegExp((0, _fp.escapeRegExp)(fieldName) + '(\\.|$)');
  return !dataView.fields.getByName(fieldName) && !!dataView.fields.getAll().find(patternField => {
    var _subTypeNested$nested;
    // We only want to match a full path segment
    const subTypeNested = (0, _esQuery.getDataViewFieldSubtypeNested)(patternField);
    if (!subTypeNested) {
      return false;
    }
    return nestedRootRegex.test((_subTypeNested$nested = subTypeNested === null || subTypeNested === void 0 ? void 0 : subTypeNested.nested.path) !== null && _subTypeNested$nested !== void 0 ? _subTypeNested$nested : '');
  });
}