"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeSearchEmbeddableApi = void 0;
var _lodash = require("lodash");
var _reactFastCompare = _interopRequireDefault(require("react-fast-compare"));
var _rxjs = require("rxjs");
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _constants = require("../../common/embeddable/constants");
var _get_search_embeddable_defaults = require("./get_search_embeddable_defaults");
var _get_esql_data_view = require("../application/main/state_management/utils/get_esql_data_view");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const initializeSearchSource = async (discoverServices, serializedSearchSource) => {
  const [searchSource, parentSearchSource] = await Promise.all([discoverServices.data.search.searchSource.create(serializedSearchSource), discoverServices.data.search.searchSource.create()]);
  searchSource.setParent(parentSearchSource);
  const query = searchSource.getField('query');
  let dataView = searchSource.getField('index');
  if ((0, _esQuery.isOfAggregateQueryType)(query)) {
    dataView = await (0, _get_esql_data_view.getEsqlDataView)(query, dataView, discoverServices);
  }
  return {
    searchSource,
    dataView
  };
};
const initializedSavedSearch = (stateManager, searchSource, discoverServices) => {
  return {
    ...Object.keys(stateManager).reduce((prev, key) => {
      return {
        ...prev,
        [key]: stateManager[key].getValue()
      };
    }, discoverServices.savedSearch.getNew()),
    searchSource
  };
};
const getProjectRoutingOverrides = query => {
  if ((0, _esQuery.isOfAggregateQueryType)(query)) {
    const value = (0, _esqlUtils.getProjectRoutingFromEsqlQuery)(query.esql);
    return value ? [{
      value
    }] : undefined;
  }
};
const initializeSearchEmbeddableApi = async (initialState, {
  discoverServices
}) => {
  /** We **must** have a search source, so start by initializing it  */
  const {
    searchSource,
    dataView
  } = await initializeSearchSource(discoverServices, initialState.serializedSearchSource);
  const searchSource$ = new _rxjs.BehaviorSubject(searchSource);
  const dataViews$ = new _rxjs.BehaviorSubject(dataView ? [dataView] : undefined);
  const defaults = (0, _get_search_embeddable_defaults.getSearchEmbeddableDefaults)(discoverServices.uiSettings);

  /** This is the state that can be initialized from the saved initial state */
  const columns$ = new _rxjs.BehaviorSubject(initialState.columns);
  const grid$ = new _rxjs.BehaviorSubject(initialState.grid);
  const headerRowHeight$ = new _rxjs.BehaviorSubject(initialState.headerRowHeight);
  const rowHeight$ = new _rxjs.BehaviorSubject(initialState.rowHeight);
  const rowsPerPage$ = new _rxjs.BehaviorSubject(initialState.rowsPerPage);
  const sampleSize$ = new _rxjs.BehaviorSubject(initialState.sampleSize);
  const density$ = new _rxjs.BehaviorSubject(initialState.density);
  const sort$ = new _rxjs.BehaviorSubject(initialState.sort);
  const savedSearchViewMode$ = new _rxjs.BehaviorSubject(initialState.viewMode);

  /**
   * This is the state that comes from the search source that needs individual publishing subjects for the API
   * - Note that these subjects can't currently be changed on their own, and therefore we do not need to keep
   *   them "in sync" with changes to the search source. This would change with inline editing.
   */
  const filters$ = new _rxjs.BehaviorSubject(searchSource.getField('filter'));
  const query$ = new _rxjs.BehaviorSubject(searchSource.getField('query'));
  const initialQuery = searchSource.getField('query');
  const projectRoutingOverrides$ = new _rxjs.BehaviorSubject(getProjectRoutingOverrides(initialQuery));
  const canEditUnifiedSearch = () => false;

  /** This is the state that has to be fetched */
  const rows$ = new _rxjs.BehaviorSubject([]);
  const columnsMeta$ = new _rxjs.BehaviorSubject(undefined);
  const totalHitCount$ = new _rxjs.BehaviorSubject(undefined);
  const inspectorAdapters$ = new _rxjs.BehaviorSubject({});

  /**
   * The state manager is used to modify the state of the saved search - this should never be
   * treated as the source of truth
   */
  const stateManager = {
    columns: columns$,
    columnsMeta: columnsMeta$,
    grid: grid$,
    headerRowHeight: headerRowHeight$,
    rows: rows$,
    rowHeight: rowHeight$,
    rowsPerPage: rowsPerPage$,
    sampleSize: sampleSize$,
    sort: sort$,
    totalHitCount: totalHitCount$,
    viewMode: savedSearchViewMode$,
    density: density$,
    inspectorAdapters: inspectorAdapters$
  };

  /** The saved search should be the source of truth for all state  */
  const savedSearch$ = new _rxjs.BehaviorSubject(initializedSavedSearch(stateManager, searchSource, discoverServices));

  /** This will fire when any of the **editable** state changes */
  const onAnyStateChange = (0, _rxjs.combineLatest)((0, _lodash.pick)(stateManager, _constants.EDITABLE_SAVED_SEARCH_KEYS));

  /** APIs for updating search source properties */
  const setDataViews = nextDataViews => {
    searchSource.setField('index', nextDataViews[0]);
    dataViews$.next(nextDataViews);
    searchSource$.next(searchSource);
  };
  const setFilters = filters => {
    searchSource.setField('filter', filters);
    filters$.next(filters);
    searchSource$.next(searchSource);
  };
  const setQuery = query => {
    searchSource.setField('query', query);
    query$.next(query);
    searchSource$.next(searchSource);
  };
  const setColumns = columns => {
    stateManager.columns.next(columns);
  };

  /** Keep the saved search in sync with any state changes */
  const syncSavedSearch = (0, _rxjs.combineLatest)([onAnyStateChange, searchSource$]).pipe((0, _rxjs.skip)(1), (0, _rxjs.map)(([newState, newSearchSource]) => ({
    ...savedSearch$.getValue(),
    ...newState,
    searchSource: newSearchSource
  }))).subscribe(newSavedSearch => {
    savedSearch$.next(newSavedSearch);
  });

  /** Keep projectRoutingOverrides$ in sync with query$ changes */
  const syncProjectRoutingOverrides = query$.subscribe(query => {
    const currentOverrides = projectRoutingOverrides$.getValue();
    const nextOverrides = getProjectRoutingOverrides(query);
    if (!(0, _reactFastCompare.default)(currentOverrides, nextOverrides)) {
      projectRoutingOverrides$.next(nextOverrides);
    }
  });
  return {
    cleanup: () => {
      syncSavedSearch.unsubscribe();
      syncProjectRoutingOverrides.unsubscribe();
    },
    api: {
      setDataViews,
      dataViews$,
      savedSearch$,
      filters$,
      setFilters,
      query$,
      setQuery,
      projectRoutingOverrides$,
      canEditUnifiedSearch,
      setColumns
    },
    stateManager,
    anyStateChange$: onAnyStateChange.pipe((0, _rxjs.map)(() => undefined)),
    comparators: {
      sort: (a, b) => (0, _reactFastCompare.default)(a !== null && a !== void 0 ? a : [], b !== null && b !== void 0 ? b : []),
      columns: 'deepEquality',
      grid: (a, b) => (0, _reactFastCompare.default)(a !== null && a !== void 0 ? a : {}, b !== null && b !== void 0 ? b : {}),
      sampleSize: (a, b) => (a !== null && a !== void 0 ? a : defaults.sampleSize) === (b !== null && b !== void 0 ? b : defaults.sampleSize),
      rowsPerPage: (a, b) => (a !== null && a !== void 0 ? a : defaults.rowsPerPage) === (b !== null && b !== void 0 ? b : defaults.rowsPerPage),
      rowHeight: (a, b) => (a !== null && a !== void 0 ? a : defaults.rowHeight) === (b !== null && b !== void 0 ? b : defaults.rowHeight),
      headerRowHeight: (a, b) => (a !== null && a !== void 0 ? a : defaults.headerRowHeight) === (b !== null && b !== void 0 ? b : defaults.headerRowHeight),
      serializedSearchSource: 'referenceEquality',
      viewMode: 'referenceEquality',
      density: 'referenceEquality'
    },
    reinitializeState: lastSaved => {
      sort$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.sort);
      columns$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.columns);
      grid$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.grid);
      sampleSize$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.sampleSize);
      rowsPerPage$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.rowsPerPage);
      rowHeight$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.rowHeight);
      headerRowHeight$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.headerRowHeight);
      savedSearchViewMode$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.viewMode);
      density$.next(lastSaved === null || lastSaved === void 0 ? void 0 : lastSaved.density);
    }
  };
};
exports.initializeSearchEmbeddableApi = initializeSearchEmbeddableApi;