"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataStreamsStatsClient = void 0;
var _ioTsUtils = require("@kbn/io-ts-utils");
var _rison = _interopRequireDefault(require("@kbn/rison"));
var _api_types = require("../../../common/api_types");
var _constants = require("../../../common/constants");
var _integration = require("../../../common/data_streams_stats/integration");
var _errors = require("../../../common/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DataStreamsStatsClient {
  constructor(http, telemetryClient) {
    this.http = http;
    this.telemetryClient = telemetryClient;
  }
  async getDataStreamsTypesPrivileges(params) {
    const types = 'types' in params ? _rison.default.encodeArray(params.types.length === 0 ? _constants.KNOWN_TYPES : params.types) : undefined;
    const response = await this.http.get('/internal/dataset_quality/data_streams/types_privileges', {
      query: {
        ...params,
        types
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data streams types privileges: ${error}`, error);
    });
    const {
      datasetTypesPrivileges
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamsTypesPrivilegesResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data streams types privileges: ${message}`))(response);
    return {
      datasetTypesPrivileges
    };
  }
  async getDataStreamsStats(params) {
    const types = 'types' in params ? _rison.default.encodeArray(params.types.length === 0 ? _constants.KNOWN_TYPES : params.types) : undefined;
    const response = await this.http.get('/internal/dataset_quality/data_streams/stats', {
      query: {
        ...params,
        types
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data streams stats: ${error}`, error);
    });
    const {
      dataStreamsStats,
      datasetUserPrivileges
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamsStatsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data streams stats response: ${message}`))(response);
    return {
      dataStreamsStats,
      datasetUserPrivileges
    };
  }
  async getDataStreamsTotalDocs(params) {
    const response = await this.http.get('/internal/dataset_quality/data_streams/total_docs', {
      query: {
        ...params
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data streams total docs: ${error}`, error);
    });
    const {
      totalDocs
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamTotalDocsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data streams total docs stats response: ${message}`))(response);
    return totalDocs;
  }
  async getDataStreamsDegradedStats(params) {
    const response = await this.http.get('/internal/dataset_quality/data_streams/degraded_docs', {
      query: {
        ...params,
        types: _rison.default.encodeArray(params.types)
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data streams degraded stats: ${error}`, error);
    });
    const {
      degradedDocs
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamDegradedDocsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data streams degraded docs stats response: ${message}`))(response);
    return degradedDocs;
  }
  async getDataStreamsFailedStats(params) {
    const response = await this.http.get('/internal/dataset_quality/data_streams/failed_docs', {
      query: {
        ...params,
        types: _rison.default.encodeArray(params.types)
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data streams failed stats: ${error}`, error);
    });
    const {
      failedDocs
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamFailedDocsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data streams failed docs stats response: ${message}`))(response);
    return failedDocs;
  }
  async getNonAggregatableDatasets(params) {
    const response = await this.http.get('/internal/dataset_quality/data_streams/non_aggregatable', {
      query: {
        ...params,
        types: _rison.default.encodeArray(params.types)
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch non aggregatable datasets: ${error}`, error);
    });
    const nonAggregatableDatasets = (0, _ioTsUtils.decodeOrThrow)(_api_types.getNonAggregatableDatasetsRt, message => new _errors.DatasetQualityError(`Failed to fetch non aggregatable datasets: ${message}`))(response);
    return nonAggregatableDatasets;
  }
  async getIntegrations() {
    const response = await this.http.get('/internal/dataset_quality/integrations').catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch integrations: ${error}`, error);
    });
    const {
      integrations
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getIntegrationsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode integrations response: ${message}`))(response);
    return integrations.map(_integration.Integration.create);
  }
  async updateFailureStore({
    dataStream,
    failureStoreEnabled,
    customRetentionPeriod
  }) {
    var _this$telemetryClient;
    const response = await this.http.put(`/internal/dataset_quality/data_streams/${dataStream}/update_failure_store`, {
      body: JSON.stringify({
        failureStoreEnabled,
        customRetentionPeriod
      })
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to update failure store": ${error}`, error);
    });
    (_this$telemetryClient = this.telemetryClient) === null || _this$telemetryClient === void 0 ? void 0 : _this$telemetryClient.trackFailureStoreUpdated({
      data_stream_name: dataStream,
      failure_store_enabled: failureStoreEnabled,
      custom_retention_period: customRetentionPeriod
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.updateFailureStoreResponseRt, message => new _errors.DatasetQualityError(`Failed to decode update failure store response: ${message}"`))(response);
  }
}
exports.DataStreamsStatsClient = DataStreamsStatsClient;