"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createFiltersFromValueClickAction = exports.createFilterESQL = exports.createFilter = exports.appendFilterToESQLQueryFromValueClickAction = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _public = require("@kbn/expressions-plugin/public");
var _esqlUtils = require("@kbn/esql-utils");
var _esQuery = require("@kbn/es-query");
var _build_filters = require("@kbn/es-query/src/filters/build_filters");
var _fieldFormatsCommon = require("@kbn/field-formats-common");
var _services = require("../../services");
var _query = require("../../query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * For terms aggregations on `__other__` buckets, this assembles a list of applicable filter
 * terms based on a specific cell in the tabified data.
 *
 * @param  {EventData['table']} table - tabified table data
 * @param  {number} columnIndex - current column index
 * @param  {number} rowIndex - current row index
 * @return {array} - array of terms to filter against
 */
const getOtherBucketFilterTerms = (table, columnIndex, rowIndex) => {
  if (rowIndex === -1) {
    return [];
  }

  // get only rows where cell value matches current row for all the fields before columnIndex
  const rows = table.rows.filter(row => {
    return table.columns.every((column, i) => {
      return row[column.id] === table.rows[rowIndex][column.id] || i >= columnIndex;
    });
  });
  const terms = rows.map(row => row[table.columns[columnIndex].id]);
  return [...new Set(terms.filter(term => {
    const notOther = String(term) !== '__other__';
    const notMissing = String(term) !== _fieldFormatsCommon.MISSING_TOKEN;
    return notOther && notMissing;
  }))];
};

/**
 * Assembles the filters needed to apply filtering against a specific cell value, while accounting
 * for cases like if the value is a terms agg in an `__other__` or `__missing__` bucket.
 *
 * @param  {EventData['table']} table - tabified table data
 * @param  {number} columnIndex - current column index
 * @param  {number} rowIndex - current row index
 * @param  {string} cellValue - value of the current cell
 * @return {Filter[]|undefined} - list of filters to provide to queryFilter.addFilters()
 */
const createFilter = async (table, columnIndex, rowIndex) => {
  var _table$columns$column;
  if (!table || !table.columns || !table.columns[columnIndex] || !table.columns[columnIndex].meta || table.columns[columnIndex].meta.source !== 'esaggs' || !((_table$columns$column = table.columns[columnIndex].meta.sourceParams) !== null && _table$columns$column !== void 0 && _table$columns$column.indexPatternId)) {
    return;
  }
  const column = table.columns[columnIndex];
  const {
    indexPatternId,
    ...aggConfigParams
  } = table.columns[columnIndex].meta.sourceParams;
  const aggConfigsInstance = (0, _services.getSearchService)().aggs.createAggConfigs(await (0, _services.getIndexPatterns)().get(indexPatternId), [aggConfigParams]);
  const aggConfig = aggConfigsInstance.aggs[0];
  let filter = [];
  const value = rowIndex > -1 ? table.rows[rowIndex][column.id] : null;
  if (value === null || value === undefined || !aggConfig.isFilterable()) {
    return;
  }
  if ((aggConfig.type.name === 'terms' || aggConfig.type.name === 'multi_terms') && aggConfig.params.otherBucket) {
    const terms = getOtherBucketFilterTerms(table, columnIndex, rowIndex);
    filter = aggConfig.createFilter(value, {
      terms
    });
  } else {
    filter = aggConfig.createFilter(value);
  }
  if (!filter) {
    return;
  }
  if (!Array.isArray(filter)) {
    filter = [filter];
  }
  return filter;
};
exports.createFilter = createFilter;
const createFilterFromRawColumnsESQL = async (column, value) => {
  var _column$meta, _column$meta$sourcePa, _column$meta3;
  const indexPattern = column === null || column === void 0 ? void 0 : (_column$meta = column.meta) === null || _column$meta === void 0 ? void 0 : (_column$meta$sourcePa = _column$meta.sourceParams) === null || _column$meta$sourcePa === void 0 ? void 0 : _column$meta$sourcePa.indexPattern;
  if (!indexPattern) {
    return [];
  }
  const dataView = await (0, _esqlUtils.getESQLAdHocDataview)({
    query: 'FROM ' + indexPattern,
    dataViewsService: (0, _services.getIndexPatterns)()
  });
  const field = dataView.getFieldByName(column.name);

  // Field should be present in the data view and filterable
  if (!field || !field.filterable) {
    return [];
  }
  // Match phrase or phrases filter based on whether value is an array
  // The advantage of match_phrase is that you get a term query when it's not a text and
  // match phrase if it is a text. So you don't have to worry about the field type.
  if (Array.isArray(value)) {
    var _column$meta2;
    return [(0, _build_filters.buildPhrasesFilter)({
      name: column.name,
      type: (_column$meta2 = column.meta) === null || _column$meta2 === void 0 ? void 0 : _column$meta2.type
    }, value, dataView)];
  }
  return [(0, _build_filters.buildPhraseFilter)({
    name: column.name,
    type: (_column$meta3 = column.meta) === null || _column$meta3 === void 0 ? void 0 : _column$meta3.type
  }, value, dataView)];
};
const createFilterESQL = async (table, columnIndex, rowIndex) => {
  var _table$columns, _column$meta4, _column$meta4$sourceP, _column$meta$sourcePa2;
  const column = table === null || table === void 0 ? void 0 : (_table$columns = table.columns) === null || _table$columns === void 0 ? void 0 : _table$columns[columnIndex];
  if (!(column !== null && column !== void 0 && (_column$meta4 = column.meta) !== null && _column$meta4 !== void 0 && (_column$meta4$sourceP = _column$meta4.sourceParams) !== null && _column$meta4$sourceP !== void 0 && _column$meta4$sourceP.sourceField) || ((_column$meta$sourcePa2 = column.meta.sourceParams) === null || _column$meta$sourcePa2 === void 0 ? void 0 : _column$meta$sourcePa2.sourceField) === '___records___') {
    return [];
  }
  const sourceParams = column.meta.sourceParams;
  if (!(0, _public.isSourceParamsESQL)(sourceParams)) {
    return [];
  }
  const {
    indexPattern,
    sourceField,
    operationType,
    interval
  } = sourceParams;
  const value = rowIndex > -1 ? table.rows[rowIndex][column.id] : null;
  if (value == null) {
    return [];
  }
  const filters = [];
  if (typeof operationType === 'string' && ['date_histogram', 'histogram'].includes(operationType)) {
    filters.push((0, _build_filters.buildSimpleNumberRangeFilter)(sourceField, operationType === 'date_histogram' ? 'date' : 'number', {
      gte: value,
      lt: value + (interval !== null && interval !== void 0 ? interval : 0),
      ...(operationType === 'date_histogram' ? {
        format: 'strict_date_optional_time'
      } : {})
    }, value, indexPattern));
  } else if (!operationType) {
    filters.push(...(await createFilterFromRawColumnsESQL(column, value)));
  } else {
    filters.push((0, _build_filters.buildSimpleExistFilter)(sourceField, indexPattern));
  }
  return filters;
};

/** @public */
exports.createFilterESQL = createFilterESQL;
const createFiltersFromValueClickAction = async ({
  data,
  negate
}) => {
  const filters = [];
  for (const value of data) {
    var _table$meta, _await$createFilter;
    if (!value) {
      continue;
    }
    const {
      table,
      column,
      row
    } = value;
    const filter = ((_table$meta = table.meta) === null || _table$meta === void 0 ? void 0 : _table$meta.type) === 'es_ql' ? await createFilterESQL(table, column, row) : (_await$createFilter = await createFilter(table, column, row)) !== null && _await$createFilter !== void 0 ? _await$createFilter : [];
    filter.forEach(f => {
      if (negate) {
        f = (0, _esQuery.toggleFilterNegated)(f);
      }
      filters.push(f);
    });
  }
  return _lodash.default.uniqWith((0, _query.mapAndFlattenFilters)(filters), (a, b) => (0, _esQuery.compareFilters)(a, b, _esQuery.COMPARE_ALL_OPTIONS));
};
exports.createFiltersFromValueClickAction = createFiltersFromValueClickAction;
function getOperationForWhere(value, negate) {
  if (value == null) {
    return negate ? 'is_not_null' : 'is_null';
  }
  return negate ? '-' : '+';
}

/** @public */
const appendFilterToESQLQueryFromValueClickAction = ({
  data,
  query,
  negate
}) => {
  if (!query) {
    return;
  }
  // Do not append in case of time series, for now. We need to find a way to compute the interval
  // to create the time range filter correctly. The users can brush to update the time filter instead.
  const dataPoints = data.filter(point => {
    var _point$table, _point$table$columns, _point$table$columns$, _point$table$columns$2;
    return point && ((_point$table = point.table) === null || _point$table === void 0 ? void 0 : (_point$table$columns = _point$table.columns) === null || _point$table$columns === void 0 ? void 0 : (_point$table$columns$ = _point$table$columns[point.column]) === null || _point$table$columns$ === void 0 ? void 0 : (_point$table$columns$2 = _point$table$columns$.meta) === null || _point$table$columns$2 === void 0 ? void 0 : _point$table$columns$2.type) !== 'date';
  });
  if (!dataPoints.length) {
    return;
  }
  let queryString = query.esql;
  for (const point in dataPoints) {
    if (dataPoints[point]) {
      var _table$columns2;
      const {
        table,
        column: columnIndex,
        row: rowIndex
      } = dataPoints[point];
      if (table !== null && table !== void 0 && (_table$columns2 = table.columns) !== null && _table$columns2 !== void 0 && _table$columns2[columnIndex]) {
        var _column$meta5;
        const column = table.columns[columnIndex];
        const value = rowIndex > -1 ? table.rows[rowIndex][column.id] : null;
        const queryWithWhere = (0, _esqlUtils.appendWhereClauseToESQLQuery)(queryString, column.name, value, getOperationForWhere(value, negate || false), (_column$meta5 = column.meta) === null || _column$meta5 === void 0 ? void 0 : _column$meta5.type);
        if (queryWithWhere) {
          queryString = queryWithWhere;
        }
      }
    }
  }
  return queryString;
};
exports.appendFilterToESQLQueryFromValueClickAction = appendFilterToESQLQueryFromValueClickAction;