"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.hasAnyDefaultLLMConnectors = hasAnyDefaultLLMConnectors;
exports.updateDefaultLLMActions = updateDefaultLLMActions;
var _default_llm_connectors = require("./default_llm_connectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Checks if any default LLM connectors exist.
 */
async function hasAnyDefaultLLMConnectors(actionsClient) {
  const existingConnectors = await actionsClient.getAll();
  for (const connector of existingConnectors) {
    var _connector$config, _connector$config2;
    if (connector.actionTypeId === '.inference' && (connector === null || connector === void 0 ? void 0 : (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.taskType) === 'chat_completion' && (connector === null || connector === void 0 ? void 0 : (_connector$config2 = connector.config) === null || _connector$config2 === void 0 ? void 0 : _connector$config2.provider) === 'elastic') {
      return true;
    }
  }
  return false;
}

/**
 * This function is called when EIS (Elastic Inference Service) is being enabled. It checks
 * if the user has the `actions.save` Kibana privilege, and if so, compares the existing
 * connectors against the default LLM connectors defined in `default_llm_connectors.json`.
 * Any missing connectors are created automatically.
 */
async function updateDefaultLLMActions(getStartServices, request, logger) {
  var _capabilities$actions;
  const [coreStart, pluginsStart] = await getStartServices();
  const capabilities = await coreStart.capabilities.resolveCapabilities(request, {
    capabilityPath: 'actions.*'
  });
  const hasActionsSavePrivilege = ((_capabilities$actions = capabilities.actions) === null || _capabilities$actions === void 0 ? void 0 : _capabilities$actions.save) === true;
  if (!hasActionsSavePrivilege) {
    logger.warn('Not enough privileges to update default llm actions..');
    return;
  }
  const actionsClient = await pluginsStart.actions.getActionsClientWithRequest(request);

  // If any default LLM connector already exists, we assume they have all been created
  if (await hasAnyDefaultLLMConnectors(actionsClient)) {
    logger.info('Default LLM connectors already exist, skipping creation');
    return;
  }

  // Get all existing connectors to check by ID
  const existingConnectors = await actionsClient.getAll();
  const existingConnectorIds = new Set(existingConnectors.map(c => c.id));

  // Create any missing default LLM connectors
  for (const connector of _default_llm_connectors.defaultLLMConnectors) {
    if (existingConnectorIds.has(connector.id)) {
      logger.debug(`Default LLM connector "${connector.name}" already exists, skipping`);
      continue;
    }
    try {
      await actionsClient.create({
        action: {
          actionTypeId: connector.actionTypeId,
          name: connector.name,
          config: connector.config,
          secrets: {}
        },
        options: {
          id: connector.id
        }
      });
      logger.info(`Created default LLM connector: ${connector.name}`);
    } catch (error) {
      logger.warn(`Failed to create default LLM connector "${connector.name}": ${error.message}`);
    }
  }
}