"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCloudConnectLicenseSync = registerCloudConnectLicenseSync;
var _server = require("@kbn/core/server");
var _constants = require("../../common/constants");
var _cloud_connect_client = require("../services/cloud_connect_client");
var _storage = require("../services/storage");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Subscribe to local license changes and report them to Cloud Connect.
 * Returns the RxJS subscription so callers can unsubscribe during plugin stop.
 */
function registerCloudConnectLicenseSync({
  savedObjects,
  elasticsearchClient,
  encryptedSavedObjects,
  licensing,
  logger,
  cloudApiUrl
}) {
  const encryptedSavedObjectsClient = encryptedSavedObjects.getClient({
    includedHiddenTypes: [_constants.CLOUD_CONNECT_API_KEY_TYPE]
  });
  const savedObjectsClient = new _server.SavedObjectsClient(savedObjects.createInternalRepository([_constants.CLOUD_CONNECT_API_KEY_TYPE]));
  const storageService = new _storage.StorageService({
    encryptedSavedObjectsClient,
    savedObjectsClient,
    logger
  });
  const cloudConnectClient = new _cloud_connect_client.CloudConnectClient(logger, cloudApiUrl);
  let cachedClusterInfo;
  let clusterInfoPromise;
  const getClusterVersion = async () => {
    if (cachedClusterInfo) return cachedClusterInfo;
    if (!clusterInfoPromise) {
      clusterInfoPromise = elasticsearchClient.info().then(info => {
        cachedClusterInfo = {
          name: info.name,
          id: info.cluster_uuid,
          version: info.version.number
        };
        return cachedClusterInfo;
      }).catch(error => {
        logger.warn('Failed to fetch cluster version for Cloud Connect license sync', {
          error
        });
        throw error;
      });
    }
    return clusterInfoPromise;
  };
  return licensing.license$.subscribe({
    next: async ({
      type,
      uid
    }) => {
      try {
        const apiKeyData = await storageService.getApiKey();
        if (!apiKeyData) {
          // Cluster isn't connected yet.
          return;
        }
        const clusterInfo = await getClusterVersion();
        const license = {
          type: String(type),
          uid: String(uid)
        };
        await cloudConnectClient.updateCluster(apiKeyData.apiKey, apiKeyData.clusterId, {
          license,
          self_managed_cluster: clusterInfo
        });
      } catch (error) {
        logger.warn('Failed to sync license to Cloud Connect', {
          error
        });
      }
    }
  });
}