"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLinkedChildrenOfSpan = getLinkedChildrenOfSpan;
exports.getSpanLinksCountById = getSpanLinksCountById;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchLinkedChildrenOfSpan({
  traceId,
  apmEventClient,
  start,
  end,
  spanId
}) {
  const {
    startWithBuffer,
    endWithBuffer
  } = (0, _utils2.getBufferedTimerange)({
    start,
    end
  });
  const requiredFields = (0, _as_mutable_array.asMutableArray)([_apm.TRACE_ID]);
  const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.TRANSACTION_ID, _apm.OTEL_SPAN_LINKS_SPAN_ID, _apm.OTEL_SPAN_LINKS_TRACE_ID]);
  const response = await apmEventClient.search('fetch_linked_children_of_span', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    _source: [_apm.SPAN_LINKS],
    fields: [...requiredFields, ...optionalFields],
    track_total_hits: false,
    size: 1000,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(startWithBuffer, endWithBuffer), {
          bool: {
            minimum_should_match: 1,
            should: [...(0, _server.termQuery)(_apm.SPAN_LINKS_TRACE_ID, traceId), ...(0, _server.termQuery)(_apm.OTEL_SPAN_LINKS_TRACE_ID, traceId)]
          }
        }, ...(spanId ? [{
          bool: {
            minimum_should_match: 1,
            should: [...(0, _server.termQuery)(_apm.SPAN_LINKS_SPAN_ID, spanId), ...(0, _server.termQuery)(_apm.OTEL_SPAN_LINKS_SPAN_ID, spanId)]
          }
        }] : [])]
      }
    }
  }, {
    skipProcessorEventFilter: true
  });
  const linkedChildren = response.hits.hits.map(hit => {
    var _source$span$links, _source$span;
    const source = 'span' in hit._source ? hit._source : undefined;
    const event = (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(requiredFields).build();
    return {
      ...event,
      [_apm.SPAN_LINKS]: (_source$span$links = source === null || source === void 0 ? void 0 : (_source$span = source.span) === null || _source$span === void 0 ? void 0 : _source$span.links) !== null && _source$span$links !== void 0 ? _source$span$links : (0, _utils2.mapOtelToSpanLink)({
        trace_id: event[_apm.OTEL_SPAN_LINKS_TRACE_ID],
        span_id: event[_apm.OTEL_SPAN_LINKS_SPAN_ID]
      })
    };
  });
  // Filter out documents that don't have any span.links that match the combination of traceId and spanId
  return linkedChildren.filter(linkedChild => linkedChild[_apm.SPAN_LINKS].some(spanLink => spanLink.trace.id === traceId && (spanId ? spanLink.span.id === spanId : true)));
}
function getSpanId(linkedChild) {
  var _linkedChild$SPAN_ID;
  return (_linkedChild$SPAN_ID = linkedChild[_apm.SPAN_ID]) !== null && _linkedChild$SPAN_ID !== void 0 ? _linkedChild$SPAN_ID : linkedChild[_apm.TRANSACTION_ID];
}
async function getSpanLinksCountById({
  traceId,
  apmEventClient,
  start,
  end
}) {
  const linkedChildren = await fetchLinkedChildrenOfSpan({
    traceId,
    apmEventClient,
    start,
    end
  });
  return linkedChildren.reduce((acc, item) => {
    item[_apm.SPAN_LINKS].forEach(link => {
      // Ignores span links that don't belong to this trace
      if (link.trace.id === traceId) {
        acc[link.span.id] = (acc[link.span.id] || 0) + 1;
      }
    });
    return acc;
  }, {});
}
async function getLinkedChildrenOfSpan({
  traceId,
  spanId,
  apmEventClient,
  start,
  end
}) {
  const linkedChildren = await fetchLinkedChildrenOfSpan({
    traceId,
    spanId,
    apmEventClient,
    start,
    end
  });
  return linkedChildren.map(item => ({
    trace: {
      id: item[_apm.TRACE_ID]
    },
    span: {
      id: getSpanId(item)
    }
  }));
}