"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectionStats = getConnectionStats;
var _lodash = require("lodash");
var _join_by_key = require("../../../../common/utils/join_by_key");
var _with_apm_span = require("../../../utils/with_apm_span");
var _calculate_throughput = require("../../helpers/calculate_throughput");
var _get_destination_map = require("./get_destination_map");
var _get_stats = require("./get_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getConnectionStats({
  apmEventClient,
  start,
  end,
  numBuckets,
  filter,
  collapseBy,
  offset,
  randomSampler,
  withTimeseries = true
}) {
  return (0, _with_apm_span.withApmSpan)('get_connection_stats_and_map', async () => {
    const [allMetrics, {
      nodesBydependencyName: destinationMap,
      sampled
    }] = await Promise.all([(0, _get_stats.getStats)({
      apmEventClient,
      start,
      end,
      filter,
      numBuckets,
      offset,
      withTimeseries
    }), (0, _get_destination_map.getDestinationMap)({
      apmEventClient,
      start,
      end,
      filter,
      offset,
      randomSampler
    })]);
    const statsWithLocationIds = allMetrics.map(statsItem => {
      var _destinationMap$get;
      const {
        from,
        timeseries,
        value
      } = statsItem;
      const to = (_destinationMap$get = destinationMap.get(statsItem.to.dependencyName)) !== null && _destinationMap$get !== void 0 ? _destinationMap$get : statsItem.to;
      const location = collapseBy === 'upstream' ? from : to;
      return {
        location,
        stats: [{
          timeseries,
          value
        }],
        id: location.id
      };
    }, []);
    const statsJoinedById = (0, _join_by_key.joinByKey)(statsWithLocationIds, 'id', (a, b) => {
      const {
        stats: statsA,
        ...itemA
      } = a;
      const {
        stats: statsB,
        ...itemB
      } = b;
      return {
        ...(0, _lodash.merge)({}, itemA, itemB),
        stats: statsA.concat(statsB)
      };
    });
    const statsItems = statsJoinedById.map(item => {
      var _mergedStats$timeseri, _mergedStats$timeseri2, _mergedStats$timeseri3, _mergedStats$value$er, _mergedStats$timeseri4;
      const mergedStats = item.stats.reduce((prev, current) => {
        return {
          value: {
            latency_count: prev.value.latency_count + current.value.latency_count,
            latency_sum: prev.value.latency_sum + current.value.latency_sum,
            error_count: prev.value.error_count + current.value.error_count,
            success_count: prev.value.success_count + current.value.success_count
          },
          timeseries: prev.timeseries && current.timeseries ? (0, _join_by_key.joinByKey)([...prev.timeseries, ...current.timeseries], 'x', (a, b) => ({
            x: a.x,
            latency_count: a.latency_count + b.latency_count,
            latency_sum: a.latency_sum + b.latency_sum,
            error_count: a.error_count + b.error_count,
            success_count: a.success_count + b.success_count
          })) : undefined
        };
      }, {
        value: {
          latency_count: 0,
          latency_sum: 0,
          error_count: 0,
          success_count: 0
        },
        timeseries: []
      });
      const destStats = {
        latency: {
          value: mergedStats.value.latency_count > 0 ? mergedStats.value.latency_sum / mergedStats.value.latency_count : null,
          timeseries: (_mergedStats$timeseri = mergedStats.timeseries) === null || _mergedStats$timeseri === void 0 ? void 0 : _mergedStats$timeseri.map(point => ({
            x: point.x,
            y: point.latency_count > 0 ? point.latency_sum / point.latency_count : null
          }))
        },
        totalTime: {
          value: mergedStats.value.latency_sum,
          timeseries: (_mergedStats$timeseri2 = mergedStats.timeseries) === null || _mergedStats$timeseri2 === void 0 ? void 0 : _mergedStats$timeseri2.map(point => ({
            x: point.x,
            y: point.latency_sum
          }))
        },
        throughput: {
          value: mergedStats.value.latency_count > 0 ? (0, _calculate_throughput.calculateThroughputWithRange)({
            start,
            end,
            value: mergedStats.value.latency_count
          }) : null,
          timeseries: (_mergedStats$timeseri3 = mergedStats.timeseries) === null || _mergedStats$timeseri3 === void 0 ? void 0 : _mergedStats$timeseri3.map(point => ({
            x: point.x,
            y: point.latency_count > 0 ? (0, _calculate_throughput.calculateThroughputWithRange)({
              start,
              end,
              value: point.latency_count
            }) : null
          }))
        },
        errorRate: {
          value: mergedStats.value.error_count + mergedStats.value.success_count > 0 ? ((_mergedStats$value$er = mergedStats.value.error_count) !== null && _mergedStats$value$er !== void 0 ? _mergedStats$value$er : 0) / (mergedStats.value.error_count + mergedStats.value.success_count) : null,
          timeseries: (_mergedStats$timeseri4 = mergedStats.timeseries) === null || _mergedStats$timeseri4 === void 0 ? void 0 : _mergedStats$timeseri4.map(point => {
            var _point$error_count;
            return {
              x: point.x,
              y: mergedStats.value.error_count + mergedStats.value.success_count > 0 ? ((_point$error_count = point.error_count) !== null && _point$error_count !== void 0 ? _point$error_count : 0) / (point.error_count + point.success_count) : null
            };
          })
        }
      };
      return {
        ...item,
        stats: destStats
      };
    });
    return {
      statsItems,
      sampled
    };
  });
}