"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTraceMetrics = getTraceMetrics;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/apm-data-access-plugin/common");
var _transaction_error_rate = require("../../../lib/helpers/transaction_error_rate");
var _calculate_throughput = require("../../../lib/helpers/calculate_throughput");
var _transactions = require("../../../lib/helpers/transactions");
var _get_bucket_size = require("../../../../common/utils/get_bucket_size");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_NUMBER_OF_GROUPS = 100;
/**
 * Gets the preferred document source based on groupBy, filter, and data availability.
 *
 * Uses getDocumentSources to determine which document types have data for the given
 * filter and groupBy field. This automatically handles:
 * - ServiceTransactionMetric: Most efficient, but only has service.name, service.environment, transaction.type
 * - TransactionMetric: Has more dimensions (transaction.*, host.*, container.*, kubernetes.*, cloud.*, faas.*, etc.)
 * - TransactionEvent: Raw transaction docs, fallback when metrics don't have required fields
 */
async function getPreferredDocumentSource({
  apmDataAccessServices,
  start,
  end,
  groupBy,
  kqlFilter
}) {
  const kueryParts = [];
  if (kqlFilter) {
    kueryParts.push(kqlFilter);
  }
  kueryParts.push(`${groupBy}: *`);
  const kuery = kueryParts.join(' AND ');
  const documentSources = await apmDataAccessServices.getDocumentSources({
    start,
    end,
    kuery
  });
  const {
    bucketSize
  } = (0, _get_bucket_size.getBucketSize)({
    start,
    end,
    numBuckets: 100
  });
  const {
    source
  } = (0, _common.getPreferredBucketSizeAndDataSource)({
    sources: documentSources,
    bucketSizeInSeconds: bucketSize
  });
  return source;
}
async function getTraceMetrics({
  apmEventClient,
  apmDataAccessServices,
  start,
  end,
  kqlFilter,
  groupBy
}) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3;
  const source = await getPreferredDocumentSource({
    apmDataAccessServices,
    start,
    end,
    groupBy,
    kqlFilter
  });
  const {
    rollupInterval,
    hasDurationSummaryField
  } = source;
  const documentType = source.documentType;
  const durationField = (0, _transactions.getDurationFieldForTransactions)(documentType, hasDurationSummaryField);
  const outcomeAggs = (0, _transaction_error_rate.getOutcomeAggregation)(documentType);
  const response = await apmEventClient.search('get_trace_metrics', {
    apm: {
      sources: [{
        documentType,
        rollupInterval
      }]
    },
    size: 0,
    track_total_hits: false,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), ...(0, _server.kqlQuery)(kqlFilter)]
      }
    },
    aggs: {
      groups: {
        terms: {
          field: groupBy,
          size: MAX_NUMBER_OF_GROUPS
        },
        aggs: {
          avg_latency: {
            avg: {
              field: durationField
            }
          },
          ...outcomeAggs
        }
      }
    }
  });
  const buckets = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : (_response$aggregation3 = _response$aggregation2.groups) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const items = buckets.map(bucket => {
    var _bucket$avg_latency;
    const docCount = bucket.doc_count;
    const latencyValue = (_bucket$avg_latency = bucket.avg_latency) === null || _bucket$avg_latency === void 0 ? void 0 : _bucket$avg_latency.value;
    const latencyMs = latencyValue !== null && latencyValue !== undefined ? latencyValue / 1000 : null;
    const failureRate = (0, _transaction_error_rate.calculateFailedTransactionRate)(bucket);
    const throughput = (0, _calculate_throughput.calculateThroughputWithRange)({
      start,
      end,
      value: docCount
    });
    return {
      group: bucket.key,
      latency: latencyMs,
      throughput,
      failureRate
    };
  });
  return items;
}