"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataProviders = registerDataProviders;
var _get_error_sample_details = require("../../routes/errors/get_error_groups/get_error_sample_details");
var _time = require("../utils/time");
var _get_rollup_interval_for_time_range = require("../utils/get_rollup_interval_for_time_range");
var _get_apm_service_summary = require("../../routes/assistant_functions/get_apm_service_summary");
var _get_apm_downstream_dependencies = require("../../routes/assistant_functions/get_apm_downstream_dependencies");
var _get_services_items = require("../../routes/services/get_services/get_services_items");
var _get_apm_errors = require("../../routes/assistant_functions/get_observability_alert_details_context/get_apm_errors");
var _document_type = require("../../../common/document_type");
var _environment_filter_values = require("../../../common/environment_filter_values");
var _get_changepoints = require("../../routes/assistant_functions/get_changepoints");
var _get_trace_metrics = require("../tools/get_trace_metrics");
var _build_apm_tool_resources = require("../utils/build_apm_tool_resources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerDataProviders({
  core,
  plugins,
  logger
}) {
  const {
    observabilityAgentBuilder
  } = plugins;
  if (!observabilityAgentBuilder) {
    return;
  }
  observabilityAgentBuilder.registerDataProvider('apmServiceSummary', async ({
    request,
    serviceName,
    serviceEnvironment,
    start,
    end,
    transactionType
  }) => {
    const {
      apmEventClient,
      apmAlertsClient,
      mlClient,
      esClient
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    return (0, _get_apm_service_summary.getApmServiceSummary)({
      apmEventClient,
      esClient: esClient.asCurrentUser,
      apmAlertsClient,
      mlClient,
      logger,
      arguments: {
        'service.name': serviceName,
        'service.environment': serviceEnvironment,
        start,
        end,
        'transaction.type': transactionType
      }
    });
  });
  observabilityAgentBuilder.registerDataProvider('apmDownstreamDependencies', async ({
    request,
    serviceName,
    serviceEnvironment,
    start,
    end
  }) => {
    const {
      apmEventClient,
      randomSampler
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    return (0, _get_apm_downstream_dependencies.getApmDownstreamDependencies)({
      apmEventClient,
      randomSampler,
      arguments: {
        serviceName,
        serviceEnvironment: serviceEnvironment ? serviceEnvironment : _environment_filter_values.ENVIRONMENT_ALL.value,
        start,
        end
      }
    });
  });
  observabilityAgentBuilder.registerDataProvider('apmErrors', async ({
    request,
    serviceName,
    serviceEnvironment,
    start,
    end
  }) => {
    const {
      apmEventClient
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    return (0, _get_apm_errors.getApmErrors)({
      apmEventClient,
      serviceName,
      serviceEnvironment,
      start,
      end
    });
  });
  observabilityAgentBuilder.registerDataProvider('apmExitSpanChangePoints', async ({
    request,
    serviceName,
    serviceEnvironment,
    start,
    end
  }) => {
    const {
      apmEventClient
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    return (0, _get_changepoints.getExitSpanChangePoints)({
      apmEventClient,
      serviceName,
      serviceEnvironment,
      start,
      end
    });
  });
  observabilityAgentBuilder.registerDataProvider('apmServiceChangePoints', async ({
    request,
    serviceName,
    serviceEnvironment,
    transactionType,
    transactionName,
    start,
    end
  }) => {
    const {
      apmEventClient
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    return (0, _get_changepoints.getServiceChangePoints)({
      apmEventClient,
      serviceName,
      serviceEnvironment,
      transactionType,
      transactionName,
      start,
      end
    });
  });
  observabilityAgentBuilder.registerDataProvider('apmErrorDetails', async ({
    request,
    errorId,
    serviceName,
    serviceEnvironment,
    start,
    end,
    kuery = ''
  }) => {
    const {
      apmEventClient
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    return (0, _get_error_sample_details.getErrorSampleDetails)({
      apmEventClient,
      errorId,
      serviceName,
      start: (0, _time.parseDatemath)(start),
      end: (0, _time.parseDatemath)(end),
      environment: serviceEnvironment !== null && serviceEnvironment !== void 0 ? serviceEnvironment : '',
      kuery
    });
  });
  observabilityAgentBuilder.registerDataProvider('servicesItems', async ({
    request,
    environment,
    kuery,
    start,
    end,
    searchQuery
  }) => {
    const {
      apmEventClient,
      randomSampler,
      mlClient,
      apmAlertsClient
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    const startMs = (0, _time.parseDatemath)(start);
    const endMs = (0, _time.parseDatemath)(end);
    return (0, _get_services_items.getServicesItems)({
      apmEventClient,
      apmAlertsClient,
      randomSampler,
      mlClient,
      logger,
      environment: environment !== null && environment !== void 0 ? environment : _environment_filter_values.ENVIRONMENT_ALL.value,
      kuery: kuery !== null && kuery !== void 0 ? kuery : '',
      start: startMs,
      end: endMs,
      serviceGroup: null,
      documentType: _document_type.ApmDocumentType.TransactionMetric,
      rollupInterval: (0, _get_rollup_interval_for_time_range.getRollupIntervalForTimeRange)(startMs, endMs),
      useDurationSummary: true,
      // Note: This will not work for pre 8.7 data. See: https://github.com/elastic/kibana/issues/167578
      searchQuery
    });
  });
  observabilityAgentBuilder.registerDataProvider('traceMetrics', async ({
    request,
    start,
    end,
    kqlFilter,
    groupBy
  }) => {
    const {
      apmEventClient,
      apmDataAccessServices
    } = await (0, _build_apm_tool_resources.buildApmToolResources)({
      core,
      plugins,
      request,
      logger
    });
    const startMs = (0, _time.parseDatemath)(start);
    const endMs = (0, _time.parseDatemath)(end);
    return (0, _get_trace_metrics.getTraceMetrics)({
      apmEventClient,
      apmDataAccessServices,
      start: startMs,
      end: endMs,
      kqlFilter,
      groupBy
    });
  });
}