"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RULE_SAVED_OBJECT_TYPE = void 0;
exports.getRuleIdsWithGaps = getRuleIdsWithGaps;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _authorization = require("../../../../authorization");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _utils = require("../utils");
var _build_gaps_filter = require("../../../../lib/rule_gaps/build_gaps_filter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RULE_SAVED_OBJECT_TYPE = exports.RULE_SAVED_OBJECT_TYPE = 'alert';
/**
 * Returns rule ids that have gaps within the requested time range.
 *
 * Parameters:
 * - statuses: Direct per-gap status filter applied to event log gap documents
 *   before aggregation. This corresponds to gap-level statuses
 *   (e.g. 'unfilled' | 'partially_filled' | 'filled') and controls which
 *   gaps are considered in the aggregations and latest timestamp query.
 *
 * - highestPriorityGapFillStatuses: Computed, per-rule status filter applied after
 *   aggregation. For each rule we compute an aggregated status from the
 *   summed gap durations with precedence: unfilled > in_progress > filled.
 *   Only rules whose computed aggregated status matches one of the provided
 *   values ('unfilled' | 'in_progress' | 'filled') are returned.
 */
const MAX_RULES_TO_FETCH = 10000;
async function getRuleIdsWithGaps(context, params) {
  try {
    var _params$maxRulesToFet, _aggs$aggregations, _byRuleAgg$buckets, _aggs$aggregations2, _latestGapTimestampAg;
    let authorizationTuple;
    try {
      authorizationTuple = await context.authorization.getFindAuthorizationFilter({
        authorizationEntity: _authorization.AlertingAuthorizationEntity.Rule,
        filterOpts: {
          type: _authorization.AlertingAuthorizationFilterType.KQL,
          fieldNames: {
            ruleTypeId: 'kibana.alert.rule.rule_type_id',
            consumer: 'kibana.alert.rule.consumer'
          }
        }
      });
    } catch (error) {
      var _context$auditLogger;
      (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
        action: _audit_events.RuleAuditAction.GET_RULES_WITH_GAPS,
        error
      }));
      throw error;
    }
    const {
      start,
      end,
      statuses,
      sortOrder,
      ruleTypes,
      ruleIds: ruleIdsFilter,
      highestPriorityGapFillStatuses = []
    } = params;
    const eventLogClient = await context.getEventLogClient();
    let filter = (0, _build_gaps_filter.buildGapsFilter)({
      start,
      end,
      statuses,
      hasUnfilledIntervals: params.hasUnfilledIntervals,
      hasInProgressIntervals: params.hasInProgressIntervals,
      hasFilledIntervals: params.hasFilledIntervals
    });
    if (ruleTypes !== null && ruleTypes !== void 0 && ruleTypes.length) {
      const ruleTypesFilter = ruleTypes.map(ruleType => `(kibana.alert.rule.rule_type_id: "${ruleType.type}" AND kibana.alert.rule.consumer: "${ruleType.consumer}")`).join(' OR ');
      filter = `${filter} AND (${ruleTypesFilter})`;
    }
    if (ruleIdsFilter !== null && ruleIdsFilter !== void 0 && ruleIdsFilter.length) {
      const ruleIdsFilterKql = [...new Set(ruleIdsFilter)].map(ruleId => `rule.id: "${ruleId}"`).join(' OR ');
      filter = `${filter} AND (${ruleIdsFilterKql})`;
    }
    const perBucketAgg = sortOrder === 'desc' ? {
      newest_gap_timestamp: {
        max: {
          field: '@timestamp'
        }
      }
    } : {
      oldest_gap_timestamp: {
        min: {
          field: '@timestamp'
        }
      }
    };
    const aggs = await eventLogClient.aggregateEventsWithAuthFilter(RULE_SAVED_OBJECT_TYPE, authorizationTuple.filter, {
      filter,
      aggs: {
        latest_gap_timestamp: {
          max: {
            field: '@timestamp'
          }
        },
        by_rule: {
          terms: {
            field: 'rule.id',
            size: (_params$maxRulesToFet = params.maxRulesToFetch) !== null && _params$maxRulesToFet !== void 0 ? _params$maxRulesToFet : MAX_RULES_TO_FETCH,
            order: sortOrder === 'desc' ? {
              newest_gap_timestamp: 'desc'
            } : {
              oldest_gap_timestamp: 'asc'
            }
          },
          aggs: {
            ...perBucketAgg,
            ..._utils.RULE_GAP_AGGREGATIONS
          }
        }
      }
    });
    const byRuleAgg = (_aggs$aggregations = aggs.aggregations) === null || _aggs$aggregations === void 0 ? void 0 : _aggs$aggregations.by_rule;
    const buckets = (_byRuleAgg$buckets = byRuleAgg === null || byRuleAgg === void 0 ? void 0 : byRuleAgg.buckets) !== null && _byRuleAgg$buckets !== void 0 ? _byRuleAgg$buckets : [];
    const ruleIds = [];
    for (const b of buckets) {
      if (highestPriorityGapFillStatuses.length === 0 || (0, _utils.hasMatchedGapFillStatus)(b, highestPriorityGapFillStatuses)) {
        ruleIds.push(b.key);
      }
    }
    const latestGapTimestampAgg = (_aggs$aggregations2 = aggs.aggregations) === null || _aggs$aggregations2 === void 0 ? void 0 : _aggs$aggregations2.latest_gap_timestamp;
    const result = {
      total: ruleIds.length,
      ruleIds,
      latestGapTimestamp: (_latestGapTimestampAg = latestGapTimestampAgg.value) !== null && _latestGapTimestampAg !== void 0 ? _latestGapTimestampAg : undefined
    };
    return result;
  } catch (err) {
    const errorMessage = `Failed to find rules with gaps`;
    context.logger.error(`${errorMessage} - ${err}`);
    throw _boom.default.boomify(err, {
      message: errorMessage
    });
  }
}