"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleTask = void 0;
var _i18n = require("@kbn/i18n");
var _uuid = require("uuid");
var _server = require("@kbn/task-manager-plugin/server");
var _alert_deletion_client = require("../alert_deletion_client");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const scheduleTask = async (context, request, settings, spaceIds) => {
  try {
    const taskManager = await context.taskManagerStartPromise;
    const securityService = await context.securityService;

    // check that there are no existing alert deletion tasks for the same spaceIds
    const result = await taskManager.fetch({
      query: {
        bool: {
          filter: [{
            term: {
              'task.taskType': _alert_deletion_client.ALERT_DELETION_TASK_TYPE
            }
          }, {
            term: {
              'task.status': _server.TaskStatus.Running
            }
          }]
        }
      }
    });
    let shouldScheduleTask = false;
    if (result.docs.length > 0) {
      // get all the space IDs from currently running tasks
      const runningSpaceIds = new Set();
      for (const task of result.docs) {
        var _task$params$spaceIds;
        ((_task$params$spaceIds = task.params.spaceIds) !== null && _task$params$spaceIds !== void 0 ? _task$params$spaceIds : []).forEach(spaceId => runningSpaceIds.add(spaceId));
      }
      context.logger.debug(`Found alert deletion tasks running for space IDs: ${[...runningSpaceIds].join(', ')}`);

      // check if any of the requested space IDs are already running
      const alreadyRunning = [...runningSpaceIds].filter(spaceId => spaceIds.includes(spaceId));
      if (alreadyRunning.length > 0) {
        return _i18n.i18n.translate('xpack.alerting.alertDeletion.scheduleTask.alreadyRunningError', {
          defaultMessage: 'Alert deletion task is already running for this space.'
        });
      } else {
        shouldScheduleTask = true;
      }
    } else {
      shouldScheduleTask = true;
    }
    if (shouldScheduleTask) {
      var _context$auditService, _context$auditService2;
      await taskManager.schedule({
        id: `Alerting-${_alert_deletion_client.ALERT_DELETION_TASK_TYPE}-${(0, _uuid.v4)()}`,
        taskType: _alert_deletion_client.ALERT_DELETION_TASK_TYPE,
        scope: ['alerting'],
        state: {},
        params: {
          settings,
          spaceIds
        }
      });
      const user = securityService.authc.getCurrentUser(request);
      (_context$auditService = context.auditService) === null || _context$auditService === void 0 ? void 0 : (_context$auditService2 = _context$auditService.asScoped(request)) === null || _context$auditService2 === void 0 ? void 0 : _context$auditService2.log((0, _lib.alertAuditEvent)({
        action: _lib.AlertAuditAction.SCHEDULE_DELETE,
        outcome: 'success',
        actor: user === null || user === void 0 ? void 0 : user.username,
        bulk: true
      }));
    }
  } catch (err) {
    var _context$auditService3, _context$auditService4;
    context.logger.error(`Error scheduling alert deletion task: ${err.message}`);
    (_context$auditService3 = context.auditService) === null || _context$auditService3 === void 0 ? void 0 : (_context$auditService4 = _context$auditService3.asScoped(request)) === null || _context$auditService4 === void 0 ? void 0 : _context$auditService4.log((0, _lib.alertAuditEvent)({
      action: _lib.AlertAuditAction.SCHEDULE_DELETE,
      outcome: 'failure',
      bulk: true,
      error: err
    }));
    throw err;
  }
};
exports.scheduleTask = scheduleTask;