"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._resetFlyoutManagerStore = _resetFlyoutManagerStore;
exports.getFlyoutManagerStore = getFlyoutManagerStore;
var _actions = require("./actions");
var _reducer = require("./reducer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function createStore() {
  var initial = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : _reducer.initialState;
  var currentState = initial;
  var listeners = new Set();
  var getState = function getState() {
    return currentState;
  };
  var subscribe = function subscribe(listener) {
    listeners.add(listener);
    return function () {
      listeners.delete(listener);
    };
  };

  // The onClick handlers won't execute until after store is fully assigned.
  // eslint-disable-next-line prefer-const -- Forward declaration requires 'let' not 'const'
  var store;
  var computeHistoryItems = function computeHistoryItems() {
    var currentSessionIndex = currentState.sessions.length - 1;
    var previousSessions = currentState.sessions.slice(0, currentSessionIndex);
    return previousSessions.reverse().map(function (_ref) {
      var title = _ref.title,
        mainFlyoutId = _ref.mainFlyoutId;
      return {
        title: title,
        onClick: function onClick() {
          store.dispatch((0, _actions.goToFlyout)(mainFlyoutId));
        }
      };
    });
  };
  var dispatch = function dispatch(action) {
    var nextState = (0, _reducer.flyoutManagerReducer)(currentState, action);
    if (nextState !== currentState) {
      var previousSessions = currentState.sessions;
      currentState = nextState;

      // Recompute history items eagerly if sessions changed
      // This ensures stable references and avoids stale closures
      if (nextState.sessions !== previousSessions) {
        store.historyItems = computeHistoryItems();
      }
      listeners.forEach(function (l) {
        l();
      });
    }
  };
  store = {
    getState: getState,
    subscribe: subscribe,
    dispatch: dispatch,
    addFlyout: function addFlyout(flyoutId, title, level, size) {
      return dispatch((0, _actions.addFlyout)(flyoutId, title, level, size));
    },
    closeFlyout: function closeFlyout(flyoutId) {
      return dispatch((0, _actions.closeFlyout)(flyoutId));
    },
    setActiveFlyout: function setActiveFlyout(flyoutId) {
      return dispatch((0, _actions.setActiveFlyout)(flyoutId));
    },
    setFlyoutWidth: function setFlyoutWidth(flyoutId, width) {
      return dispatch((0, _actions.setFlyoutWidth)(flyoutId, width));
    },
    setPushPadding: function setPushPadding(side, width) {
      return dispatch((0, _actions.setPushPadding)(side, width));
    },
    goBack: function goBack() {
      return dispatch((0, _actions.goBack)());
    },
    goToFlyout: function goToFlyout(flyoutId) {
      return dispatch((0, _actions.goToFlyout)(flyoutId));
    },
    addUnmanagedFlyout: function addUnmanagedFlyout(flyoutId) {
      return dispatch((0, _actions.addUnmanagedFlyout)(flyoutId));
    },
    closeUnmanagedFlyout: function closeUnmanagedFlyout(flyoutId) {
      return dispatch((0, _actions.closeUnmanagedFlyout)(flyoutId));
    },
    historyItems: computeHistoryItems() // Initialize with current state
  };
  return store;
}

// Module-level singleton.  A necessary trade-off to avoid global namespace pollution or the need for a third-party library.
var storeInstance = null;

/**
 * Returns a singleton store instance shared across all React roots within the same JS context.
 * Uses module-level singleton to ensure deduplication even if modules are loaded twice.
 */
function getFlyoutManagerStore() {
  if (!storeInstance) {
    storeInstance = createStore();
  }
  return storeInstance;
}

/**
 * For testing purposes - allows resetting the store
 */
function _resetFlyoutManagerStore() {
  storeInstance = null;
}