/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

package org.elasticsearch.search.aggregations.bucket.histogram;

import org.apache.lucene.index.BinaryDocValues;
import org.elasticsearch.index.fielddata.FieldData;
import org.elasticsearch.index.mapper.RangeFieldMapper;
import org.elasticsearch.index.mapper.RangeType;
import org.elasticsearch.search.aggregations.AggregationExecutionContext;
import org.elasticsearch.search.aggregations.Aggregator;
import org.elasticsearch.search.aggregations.AggregatorFactories;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.CardinalityUpperBound;
import org.elasticsearch.search.aggregations.LeafBucketCollector;
import org.elasticsearch.search.aggregations.LeafBucketCollectorBase;
import org.elasticsearch.search.aggregations.support.AggregationContext;
import org.elasticsearch.search.aggregations.support.ValuesSource;
import org.elasticsearch.search.aggregations.support.ValuesSourceConfig;

import java.io.IOException;
import java.util.List;
import java.util.Map;

public class RangeHistogramAggregator extends AbstractHistogramAggregator {
    private final ValuesSource.Range valuesSource;

    public RangeHistogramAggregator(
        String name,
        AggregatorFactories factories,
        double interval,
        double offset,
        BucketOrder order,
        boolean keyed,
        long minDocCount,
        DoubleBounds extendedBounds,
        DoubleBounds hardBounds,
        ValuesSourceConfig valuesSourceConfig,
        AggregationContext context,
        Aggregator parent,
        CardinalityUpperBound cardinality,
        Map<String, Object> metadata
    ) throws IOException {
        super(
            name,
            factories,
            interval,
            offset,
            order,
            keyed,
            minDocCount,
            extendedBounds,
            hardBounds,
            valuesSourceConfig.format(),
            context,
            parent,
            cardinality,
            metadata
        );
        // TODO: Stop using nulls here
        this.valuesSource = valuesSourceConfig.hasValues() ? (ValuesSource.Range) valuesSourceConfig.getValuesSource() : null;
        if (this.valuesSource.rangeType().isNumeric() == false) {
            throw new IllegalArgumentException(
                "Expected numeric range type but found non-numeric range [" + this.valuesSource.rangeType().name + "]"
            );
        }
    }

    @Override
    protected LeafBucketCollector getLeafCollector(AggregationExecutionContext aggCtx, LeafBucketCollector sub) throws IOException {
        if (valuesSource == null) {
            return LeafBucketCollector.NO_OP_COLLECTOR;
        }
        // Is it possible for multiple values here? Multiple ranges are encoded into the same BytesRef in the binary doc values
        final BinaryDocValues values = FieldData.unwrapSingleton(valuesSource.bytesValues(aggCtx.getLeafReaderContext()));
        assert values != null : "unexpected multi-value binary doc values";
        final RangeType rangeType = valuesSource.rangeType();
        return new LeafBucketCollectorBase(sub, values) {
            @Override
            public void collect(int doc, long owningBucketOrd) throws IOException {
                if (values.advanceExact(doc)) {
                    double previousKey = Double.NEGATIVE_INFINITY;
                    final List<RangeFieldMapper.Range> ranges = rangeType.decodeRanges(values.binaryValue());
                    double previousFrom = Double.NEGATIVE_INFINITY;
                    for (RangeFieldMapper.Range range : ranges) {
                        final Double from = rangeType.doubleValue(range.getFrom());
                        // The encoding should ensure that this assert is always true.
                        assert from >= previousFrom : "Start of range not >= previous start";
                        final Double to = rangeType.doubleValue(range.getTo());
                        final double effectiveFrom = (hardBounds != null && hardBounds.getMin() != null)
                            ? Double.max(from, hardBounds.getMin())
                            : from;
                        final double effectiveTo = (hardBounds != null && hardBounds.getMax() != null)
                            ? Double.min(to, hardBounds.getMax())
                            : to;
                        final double startKey = Math.floor((effectiveFrom - offset) / interval);
                        final double endKey = Math.floor((effectiveTo - offset) / interval);
                        for (double key = Math.max(startKey, previousKey); key <= endKey; key++) {
                            if (key == previousKey) {
                                continue;
                            }
                            // Bucket collection identical to NumericHistogramAggregator, could be refactored
                            long bucketOrd = bucketOrds.add(owningBucketOrd, Double.doubleToLongBits(key));
                            if (bucketOrd < 0) { // already seen
                                bucketOrd = -1 - bucketOrd;
                                collectExistingBucket(sub, doc, bucketOrd);
                            } else {
                                collectBucket(sub, doc, bucketOrd);
                            }
                        }
                        if (endKey > previousKey) {
                            previousKey = endKey;
                        }
                    }
                }
            }
        };
    }
}
