"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCleanupBeforeExit = createCleanupBeforeExit;
var _lodash = require("lodash");
var _wrap_cleanup_callback = require("./wrap_cleanup_callback");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createCleanupBeforeExit(proc) {
  const cleanupHandlers = [];
  const processExitController = new AbortController();
  let handlersInstalled = false;
  const originalProcessExit = proc.exit.bind(proc);
  const cleanup = (0, _lodash.once)(() => {
    return Promise.allSettled(cleanupHandlers.map(handler => {
      return handler.fn();
    }));
  });
  const installHandlers = () => {
    if (handlersInstalled) {
      return;
    }
    handlersInstalled = true;
    proc.once('beforeExit', cleanup);
    proc.once('uncaughtExceptionMonitor', cleanup);
    proc.once('SIGINT', cleanup);
    proc.once('SIGTERM', cleanup);
    function quit(...args) {
      originalProcessExit(...args);
    }

    // @ts-expect-error
    proc.exit = (0, _lodash.once)((...args) => {
      if (cleanupHandlers.filter(handler => handler.blockExit).length === 0) {
        quit(...args);
      } else {
        processExitController.abort();
        cleanup().finally(() => {
          quit(...args);
        });
      }
    });
  };
  return function cleanupBeforeExit(callback, options = {}) {
    const fn = (0, _wrap_cleanup_callback.wrapCleanupCallback)(callback, processExitController.signal, options);
    installHandlers();
    const handler = {
      fn,
      blockExit: options.blockExit || false
    };
    cleanupHandlers.push(handler);
    return () => {
      (0, _lodash.remove)(cleanupHandlers, handler);
    };
  };
}