"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildOngoingAlert = void 0;
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _strip_framework_fields = require("./strip_framework_fields");
var _nanos_to_micros = require("./nanos_to_micros");
var _format_alert = require("./format_alert");
var _filter_alert_state = require("./filter_alert_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Updates an existing alert document with data from the LegacyAlert class
 * Currently only populates framework fields and not any rule type specific fields
 */

const buildOngoingAlert = ({
  alert,
  legacyAlert,
  payload,
  isImproving,
  rule,
  runTimestamp,
  timestamp,
  kibanaVersion,
  dangerouslyCreateAlertsInAllSpaces
}) => {
  var _legacyAlert$getSched, _ref, _alert$tags, _rule$ALERT_RULE_TAGS;
  // Sets array fields to empty arrays if previously reported in the existing alert
  // but not present in the payload
  (0, _format_alert.replaceEmptyAlertFields)(alert, payload);
  const cleanedPayload = (0, _strip_framework_fields.stripFrameworkFields)(payload);

  // Make sure that any alert fields that are updateable are flattened.
  const refreshableAlertFields = (0, _format_alert.replaceRefreshableAlertFields)(alert);

  // Omit fields that are overwrite-able with undefined value
  const cleanedAlert = (0, _lodash.omit)(alert, _ruleDataUtils.ALERT_SEVERITY_IMPROVING);
  const alertState = legacyAlert.getState();
  const filteredAlertState = (0, _filter_alert_state.filterAlertState)(alertState);
  const hasAlertState = Object.keys(filteredAlertState).length > 0;
  const alertUpdates = {
    // Set latest rule configuration
    ...rule,
    // Update the timestamp to reflect latest update time
    [_ruleDataUtils.TIMESTAMP]: timestamp,
    [_ruleDataUtils.EVENT_ACTION]: 'active',
    [_ruleDataUtils.ALERT_RULE_EXECUTION_TIMESTAMP]: runTimestamp !== null && runTimestamp !== void 0 ? runTimestamp : timestamp,
    // Because we're building this alert after the action execution handler has been
    // run, the scheduledExecutionOptions for the alert has been cleared and
    // the lastScheduledActions has been set. If we ever change the order of operations
    // to build and persist the alert before action execution handler, we will need to
    // update where we pull the action group from.
    // Set latest action group as this may have changed during execution (ex: error -> warning)
    [_ruleDataUtils.ALERT_ACTION_GROUP]: (_legacyAlert$getSched = legacyAlert.getScheduledActionOptions()) === null || _legacyAlert$getSched === void 0 ? void 0 : _legacyAlert$getSched.actionGroup,
    // Set latest flapping state
    [_ruleDataUtils.ALERT_FLAPPING]: legacyAlert.getFlapping(),
    // Set latest flapping_history
    [_ruleDataUtils.ALERT_FLAPPING_HISTORY]: legacyAlert.getFlappingHistory(),
    // Set latest maintenance window IDs
    [_ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS]: legacyAlert.getMaintenanceWindowIds(),
    // Set latest match count
    [_ruleDataUtils.ALERT_CONSECUTIVE_MATCHES]: legacyAlert.getActiveCount(),
    [_ruleDataUtils.ALERT_PENDING_RECOVERED_COUNT]: legacyAlert.getPendingRecoveredCount(),
    // Set the time range
    ...(alertState.start ? {
      [_ruleDataUtils.ALERT_TIME_RANGE]: {
        gte: alertState.start
      }
    } : {}),
    // Set latest duration as ongoing alerts should have updated duration
    ...(alertState.duration ? {
      [_ruleDataUtils.ALERT_DURATION]: (0, _nanos_to_micros.nanosToMicros)(alertState.duration)
    } : {}),
    ...(isImproving != null ? {
      [_ruleDataUtils.ALERT_SEVERITY_IMPROVING]: isImproving
    } : {}),
    [_ruleDataUtils.ALERT_PREVIOUS_ACTION_GROUP]: (0, _lodash.get)(alert, _ruleDataUtils.ALERT_ACTION_GROUP),
    [_ruleDataUtils.SPACE_IDS]: dangerouslyCreateAlertsInAllSpaces === true ? ['*'] : rule[_ruleDataUtils.SPACE_IDS],
    [_ruleDataUtils.VERSION]: kibanaVersion,
    [_ruleDataUtils.TAGS]: Array.from(new Set([...((_ref = cleanedPayload === null || cleanedPayload === void 0 ? void 0 : cleanedPayload.tags) !== null && _ref !== void 0 ? _ref : []), ...((_alert$tags = alert.tags) !== null && _alert$tags !== void 0 ? _alert$tags : []), ...((_rule$ALERT_RULE_TAGS = rule[_ruleDataUtils.ALERT_RULE_TAGS]) !== null && _rule$ALERT_RULE_TAGS !== void 0 ? _rule$ALERT_RULE_TAGS : [])])),
    ...(hasAlertState ? {
      [_ruleDataUtils.ALERT_STATE_NAMESPACE]: filteredAlertState
    } : {})
  };

  // Clean the existing alert document so any nested fields that will be updated
  // are removed, to avoid duplicate data.
  // e.g. if the existing alert document has the field:
  // {
  //   kibana: {
  //     alert: {
  //       field1: 'value1'
  //     }
  //   }
  // }
  // and the updated alert has the field
  // {
  //   'kibana.alert.field1': 'value2'
  // }
  // the expanded field from the existing alert is removed
  const expandedAlert = (0, _format_alert.removeUnflattenedFieldsFromAlert)(cleanedAlert, {
    ...cleanedPayload,
    ...alertUpdates,
    ...refreshableAlertFields
  });
  return _deepmerge.default.all([expandedAlert, refreshableAlertFields, cleanedPayload, alertUpdates], {
    arrayMerge: (_, sourceArray) => sourceArray
  });
};
exports.buildOngoingAlert = buildOngoingAlert;