"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecretsSchema = exports.Response = exports.FailureResponse = exports.CreateAlertParamsSchema = exports.ConfigSchema = exports.CloseAlertParamsSchema = void 0;
var _zod = require("@kbn/zod");
var _lodash = require("lodash");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ConfigSchema = exports.ConfigSchema = _zod.z.object({
  apiUrl: _zod.z.string()
}).strict();
const SecretsSchema = exports.SecretsSchema = _zod.z.object({
  apiKey: _zod.z.string()
}).strict();
const SuccessfulResponse = _zod.z.object({
  took: _zod.z.coerce.number(),
  requestId: _zod.z.string(),
  result: _zod.z.string()
}).passthrough();
const FailureResponse = exports.FailureResponse = _zod.z.object({
  code: _zod.z.coerce.number().optional(),
  message: _zod.z.string().optional(),
  result: _zod.z.string().optional(),
  errors: _zod.z.array(_zod.z.object({
    title: _zod.z.string(),
    code: _zod.z.string()
  }).strict()).optional(),
  took: _zod.z.coerce.number().optional(),
  requestId: _zod.z.string().optional()
}).passthrough();
const Response = exports.Response = _zod.z.union([SuccessfulResponse, FailureResponse]);
const responderTypes = _zod.z.enum(['team', 'user', 'escalation', 'schedule']);

/**
 * For more information on the JiraServiceManagement create alert schema see: https://developer.atlassian.com/cloud/jira/service-desk-ops/rest/v1/api-group-integration-events/#api-jsm-ops-integration-v2-alerts-post
 */
const CreateAlertParamsSchema = exports.CreateAlertParamsSchema = _zod.z.object({
  message: _zod.z.string().min(1).max(130).superRefine((message, ctx) => {
    if ((0, _lodash.isEmpty)(message.trim())) {
      ctx.addIssue({
        code: _zod.z.ZodIssueCode.custom,
        message: _constants.MESSAGE_NON_EMPTY
      });
    }
  }),
  /**
   * The max length here should be 512 according to JiraServiceManagement's docs but we will sha256 hash the alias if it is longer than 512
   * so we'll not impose a limit on the schema otherwise it'll get rejected prematurely.
   */
  alias: _zod.z.string().optional(),
  description: _zod.z.string().max(15000).optional(),
  responders: _zod.z.array(_zod.z.object({
    id: _zod.z.string(),
    type: responderTypes
  }).strict()).max(50).optional(),
  visibleTo: _zod.z.array(_zod.z.union([_zod.z.object({
    id: _zod.z.string(),
    type: _zod.z.literal('team')
  }).strict(), _zod.z.object({
    id: _zod.z.string(),
    type: _zod.z.literal('user')
  }).strict()])).max(50).optional(),
  actions: _zod.z.array(_zod.z.string().max(50)).max(10).optional(),
  tags: _zod.z.array(_zod.z.string().max(50)).max(20).optional(),
  /**
   * The validation requirement here is that the total characters between the key and value do not exceed 8000. JiraServiceManagement
   * will truncate the value if it would exceed the 8000 but it doesn't throw an error. Because of this I'm intentionally
   * not validating the length of the keys and values here.
   */
  details: _zod.z.record(_zod.z.string(), _zod.z.string()).optional(),
  entity: _zod.z.string().max(512).optional(),
  source: _zod.z.string().max(100).optional(),
  priority: _zod.z.enum(['P1', 'P2', 'P3', 'P4', 'P5']).optional(),
  note: _zod.z.string().max(25000).optional(),
  user: _zod.z.string().max(100).optional()
}).strict();
const CloseAlertParamsSchema = exports.CloseAlertParamsSchema = _zod.z.object({
  alias: _zod.z.string(),
  user: _zod.z.string().max(100).optional(),
  source: _zod.z.string().max(100).optional(),
  note: _zod.z.string().max(25000).optional()
}).strict();