"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApiKeyHeaderAuth = void 0;
var _v = require("@kbn/zod/v4");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const HEADER_FIELD_DEFAULT = 'Api-Key';
const authSchema = _v.z.object({
  headerField: _v.z.string().meta({
    sensitive: true
  }).describe('API Key header field').default(HEADER_FIELD_DEFAULT),
  apiKey: _v.z.string().meta({
    sensitive: true
  }).describe('API Key')
});
/**
 * Header-based authentication (generic)
 * Use for: API keys, custom headers (X-API-Key, etc.)
 */
const ApiKeyHeaderAuth = exports.ApiKeyHeaderAuth = {
  id: 'api_key_header',
  schema: authSchema,
  normalizeSchema: defaults => {
    const schemaToUse = _v.z.object({
      ...authSchema.shape
    });
    if (defaults) {
      // get the default values for the headerField
      const headerField = defaults.headerField && (0, _lodash.isString)(defaults.headerField) ? defaults.headerField : HEADER_FIELD_DEFAULT;
      return _v.z.object({
        [headerField]: schemaToUse.shape.apiKey
      });
    }
    return schemaToUse;
  },
  configure: (axiosInstance, secret) => {
    // set global defaults
    Object.keys(secret).filter(key => key !== 'authType').forEach(key => {
      axiosInstance.defaults.headers.common[key] = secret[key];
    });
    return axiosInstance;
  }
};